package nxm.ice.lib;

/*
  I/O Moduule JVM Controller

  State Structure at 0xFD000000
  Command/Status  at 0xFD000400
  Return  Status  at 0xFD000408
  Command Data    at 0xFD000410

*/
public class IceIOM extends IceJVM {

  public final static int sysdly=50/25;

  public static void run() {
    int beat,cmd,key,par,cmds;
    runFAT();
    reset();
    for (beat=0x1CE00000;;beat++) {
      awMem(HBT,beat);			// heartbeat monitor
      cmd = arMemSync(CFG_CMD);
      key = (cmd>>16);
      cmds = (cmd&0xFF);
      if (key==0x1ce0 && cmds>0) {
	par = cmd;			// check parity
	maddr = CFG_BUF;
        for (int i=0; i<cmds; i++) par ^= mdata;
        if (par!=arMem(CFG_PAR)) awMem(CFG_CMD,-7);
        else processCmdBuf(cmds);		// process command register
      }
    }
  }

  static void runFAT () {
    awMem(FAT,0x1CE0);
    awMem(FAT,0x1CE1);
  }

  protected static void reset () {
    awMem(MSG,0);
    awMem(CFG_CMD,0);
    awMem(0x10,arBus(0));
  }

  protected static void processCmdBuf (int cmds) {
    int timer = 0;
    int data = 0;
    int cbi = CFG_BUF;			// current cmd addr
    int cbl = CFG_BUF + (cmds<<2);	// last cmd addr
    for (;cmds>0;) {
      int code = arMem(cbi); cbi=cbi+4;
      int cmd  = code&0xF;		// 4b command
      int cnt  = (code>>4)&0xF;		// 4b count
      int top  = code>>8;		// 24b top
      int addr = top&0xFFFFFF;		// 24b addr
      int spi  = addr&0x8000;
      switch (cmd) {
      case 1: 	// RD Word
        if (spi!=0) data=rdSPI(addr);
        else        data=arBus(addr);
        awMem(CFG_RET,data);
        break;
      case 2:	// WR Word
        data=arMem(cbi); cbi=cbi+4;
        awBus(addr,data);
        break;
      case 3: 	// WR Byte
	addr=top&0xFFFF;
        data=(top>>16)&0xFF;		// 8b data
        if (spi!=0) wrSPI(addr,data);
        else        awBus(addr,data);
        break;
      case 4: 	// WRM - write multiple qwords (cnt=qwords-1)
	for (int i=0; i<=cnt; i++) {
          data=arMem(cbi); cbi=cbi+4;
          awBus(addr,data);
	}
        break;
      case 5: 	// DLY
        vdelay(top);
        break;
      case 6: 	// POLL
        int expect=arMem(cbi); cbi=cbi+4;
        if (data==expect) break;
        if (timer>=top) cmds=-1;
        else { vdelay(1000); timer=timer+1000; cbi=cbi-12; }  // back up to read
        break;
      case 7: 	// RWM - last on queue
        int waddr = arMem(cbi); cbi=cbi+4;
        int wdata = arMem(cbi); cbi=cbi+4;
        int wcnt  = arMem(cbi); cbi=cbi+4;
        int raddr = CFG_BUF;
        for (int i=0; i<wcnt; i++) {
          awBus(waddr,wdata);
          vdelay(20);
          awMem(raddr,arBus(addr));
	  raddr=raddr+4;
        }
        break;
      case 11: 	// PRT_CMD - addr has {partId:8,func:4,addr:12} 
        int mask=arMem(cbi); cbi=cbi+4;
            data=arMem(cbi); cbi=cbi+4;
	top  = (addr>>8)&0xFF00;	// put partId in page slot
	addr = addr&0xFFFF;		// has func:4|addr:12
	awBus(top|0x0,mask);
	awBus(top|0x4,0x2FFF);
	part_wait(top);
	awBus(top|0x0,data);
	awBus(top|0x4,addr);
	part_wait(top);
	break;
      default:	// Error
        awMem(CFG_CMD,-9);
	return;
      }
      cmds = (cbl-cbi)>>2;
      awMem(CFG_CMD,0x1ced0000|(cmds&0xFFFF));
    }
  }

  protected static int part_wait (int addr) {
    int i=0,timeout=5000;
    boolean part_busy;
    do { part_busy = (arBus(addr|0x4) & 0x8000) != 0; } 
    while (part_busy && ++i<timeout);
    return i;
  }

  protected static void vdelay (int usec) {
    int jmax = sysdly;
    for (int i=0; i<usec; i=i+1) {
      for (int j=0; j<jmax; j=j+1);
    }
  }

/* 
  SPI Routines
  assumes the SPI interface is 4 bytes up from the base register of this device's page
  the byte wide register map is 0=data 4=clock 5=ena  where ena is active low
*/
  protected static void wrSPI (int addr, int data) {
    int spireg = (addr&0x3F00)|0x10004;
    int spiadr = (addr&0x00FF);
    saddr = 0;
    sdata = 0x30;
    saddr = spireg;
    if ((addr&0x4000)!=0) {	// mode=2
      sdata = 0x10;
      wrSPIr(0x00);
      wrSPIr(spiadr);
      sdata = 0x30;
      sdata = 0x10;
      wrSPIr(0x60);
      wrSPIr(data);
      sdata = 0x30;
    } else {			// mode=1
      sdata = 0x10;
      wrSPIf(spiadr);
      wrSPIf(data);
      sdata = 0x30;
    }
    saddr = 0;
  }
  // write rising edge
  protected static void wrSPIr (int data) {
    for (int i=7; i>=0; i=i-1) {
      int dat = (data>>i)&0x1;
      sdata = 0x00|dat;
      sdata = 0x10|dat;
    }
  }
  // write falling edge
  protected static void wrSPIf (int data) {
    int dat=0;
    for (int i=7; i>=0; i=i-1) {
      dat = (data>>i)&0x1;
      sdata = 0x10|dat;
      sdata = 0x00|dat;
    }
    sdata = 0x10|dat;
  }

  protected static int rdSPI (int addr) {
    int spireg = (addr&0x3F00)|0x10004;
    int spiadr = (addr&0x00FF);
    int data;
    saddr = 0;
    sdata = 0x30;
    saddr = spireg;
    if ((addr&0x4000)!=0) {	// mode=2
      sdata = 0x10;
      wrSPIr(0x00);
      wrSPIr(spiadr);
      sdata = 0x30;
      sdata = 0x10;
      wrSPIr(0xA0);
      data=rdSPIr();
      sdata = 0x30;
    } else {			// mode=1
      sdata = 0x10;
      wrSPIf(spiadr);
      data=rdSPIf();
      sdata = 0x30;
    }
    saddr = 0;
    return data;
  }
  // read rising edge
  protected static int rdSPIr () {
    int data=0;
    for (int i=7; i>=0; i=i-1) {
      sdata = 0x00;
      sdata = 0x10;
      data |= (sdata&0x1)<<i;
    }
    return data;
  }
  // read falling edge
  protected static int rdSPIf () {
    int data=0;
    for (int i=7; i>=0; i=i-1) {
      sdata = 0x10;
      sdata = 0x00;
      data |= (sdata&0x1)<<i;
    }
    sdata = 0x10;
    return data;
  }

}
