/*
  Native entries for the ICE FileIO interface 

  @author Jeff Schoen 
*/
#ifdef _LINX
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <features.h>
#include <linux/hdreg.h>
#include <sys/ioctl.h>
#include <sys/syscall.h>
#ifndef __USE_GNU
#error "ASSERT: __USE_GNU should be defined by now. Bad order for include files."
#endif
#endif

#include "cdefs.h"
#include "native.h"
#include "iceioctl.h"
#include "nxm_ice_lib_FileIO.h"
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

static int fopenflags[256];

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_fopen (JNIEnv *env, jclass clazz, jstring fname, jint dir, jint flags) {
  jint fd, iofunc; const char *cfname;
  cfname = (*env)->GetStringUTFChars(env, fname, 0);
  if (dir>0) iofunc = O_RDWR|O_CREAT;
  else if (dir<0) iofunc = O_RDONLY;
  else iofunc = O_RDWR;
//  if (flags&2) iofunc = O_RDWR|O_NONBLOCK;
#if defined(_LINX)
  if (flags&4) iofunc |= O_DIRECT;
#endif
#if defined(_WIN)
  iofunc |= O_BINARY; /* O_BINARY required to read/write binary data under Windows Platform */
#endif
  fd = (jint)open (cfname,iofunc,0666);
  fopenflags[fd&0xFF]=flags;
  /* printf("Open fd=%d name=%s dir=%d flags=%x\n",fd,cfname,dir,flags); */
  (*env)->ReleaseStringUTFChars(env, fname, cfname);
  return fd;
}

JNIEXPORT void JNICALL Java_nxm_ice_lib_FileIO_fclose (JNIEnv *env, jclass clazz, jint fd) {
  close ((int)fd);
}

JNIEXPORT jlong JNICALL Java_nxm_ice_lib_FileIO_fseek (JNIEnv *env, jclass clazz, jint fd, jlong offset) {
  return (jlong) lseek ((int)fd, (off_t)offset, SEEK_SET);
}

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_fread (JNIEnv *env, jclass clazz, jint fd, jlong lbuf, jint boff, jint bytes) {
  jpointer buf = (jpointer)lbuf;
  ICEFSIOCTL icefs; jint status=0;
 if (fopenflags[fd&0xFF]&0x8) {
  icefs.func=IOCTL_READ; icefs.bufaddr=(lbuf+boff); icefs.len=bytes;
  status = (jint) ioctl (fd,_IOR('f',0x1,ICEFSIOCTL), &icefs);
  if (status==0) status = icefs.status;
 } else {
  jint status = (jint) read ((int)fd, (void *)(buf+boff), (size_t)bytes);
 }
  if (status == -1 && errno == EAGAIN) status = 0;  /* Non-blocking I/O and no data immediately available for reading */
  return status;
}

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_fwrite (JNIEnv *env, jclass clazz, jint fd, jlong lbuf, jint boff, jint bytes) {
  jpointer buf = (jpointer)lbuf;
  ICEFSIOCTL icefs; jint status=0;
 if (fopenflags[fd&0xFF]&0x8) {
  icefs.func=IOCTL_WRITE; icefs.bufaddr=(lbuf+boff); icefs.len=bytes;
  status = (jint) ioctl (fd,_IOR('f',0x1,ICEFSIOCTL), &icefs);
  if (status==0) status = icefs.status;
 } else {
  status = (jint) write ((int)fd, (void *)(buf+boff), (size_t)bytes);
 }
  return status;
}

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_freadb (JNIEnv *env, jclass clazz, jint fd, jobject obuf, jint boff, jint bytes) {
  jbyte *buf = GetArray(obuf);
  jint status = (jint) read ((int)fd, (void *)(buf+boff), (size_t)bytes);
  if (status == -1 && errno == EAGAIN) status = 0;  /* Non-blocking I/O and no data immediately available for reading */
  ReleaseArray(obuf,buf);
  return status;
}

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_fwriteb (JNIEnv *env, jclass clazz, jint fd, jobject obuf, jint boff, jint bytes) {
  jbyte *buf = GetArray(obuf);
  jint status = (jint) write ((int)fd, (void *)(buf+boff), (size_t)bytes);
  ReleaseArray(obuf,buf);
  return status;
}

JNIEXPORT jobject JNICALL Java_nxm_ice_lib_FileIO_allocBufferN (JNIEnv *env, jclass clazz,
        jobject array, jint length) {
  jint i;
  jint *a = *(jint **)array;
  jint hsz = (a[2]==64)? 3 : 4;         /* find object header size */
  jint *b = a+hsz;                      /* use Java alloced byte buffer for new array object */
  jint bp = (jint)(((jpointer)b) & 0xFFF);
  jint hoff = (0x1000 - bp)/4 - hsz;
  if (hoff<1) hoff += 0x400;            /* make sure enough room for header and pre-header offset field */
  b[hoff-1] = hoff;                     /* set pre-header offset field */
  b = b+hoff;                           /* set to header of new array object */
  for (i=0; i<4; i++) b[i]=a[i];        /* init the array reference count, class */
  b[hsz-1]=length;                      /* add specified array length */
  *(jint **)array = b;                  /* insert back in object reference wrapper */
  return array;
}

JNIEXPORT jint JNICALL Java_nxm_ice_lib_FileIO_freeBufferN (JNIEnv *env, jclass clazz,
        jobject array) {
  jint *a = *(jint **)array;
  jint hoff = *(a-1);                   /* get header offset in original malloc buffer */
  a -= hoff;                            /* back up to original allocation address */
  return 0;
}

JNIEXPORT jlong JNICALL Java_nxm_ice_lib_FileIO_fid (JNIEnv *env, jclass clazz, jint fd) {
  jlong serial_no=0;
#ifdef _LINX
  struct hd_driveid hd;
  jint ret;
  ret = ioctl (fd,HDIO_GET_IDENTITY,&hd);
  if (ret==0) sscanf(hd.serial_no,"%llx",&serial_no);
  else perror("HDSN");
#endif
  return serial_no;
}

JNIEXPORT jlong JNICALL Java_nxm_ice_lib_FileIO_malloc_buf (JNIEnv *env, jobject obj, jlong size) {
  jlong rv = (jpointer) malloc(size);
  return rv;
}

JNIEXPORT jint  JNICALL Java_nxm_ice_lib_FileIO_mfree_buf (JNIEnv *env, jobject obj, jlong handle) {
  free((void*)(jpointer) handle);
  return 0;
}


