package nxm.ice.lib;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libm.*;

import nxm.ice.prim.sourcepic;
import nxm.ice.lib.ICEPacket;
import nxm.ice.lib.SDDSPacket;
import nxm.ice.lib.VRTPacket;

/**
  Extends DataFile to handle ICE Archiver functions.

  Although it acts as one file to the caller, it may split the data stream
  into multiple output files if setMulti() or setFunc() are non-zero.

  A TableOfContents is created with the name [archive]_toc.
  A PSD is created with the name [archive]_psd.
  Individual channel files are [archive]_N where N is the KEY field of the TOC entry.

  Special qualifiers apply to files accessed by this class:

    FUNC=s	Function supports MULTI
    MULTI=n	Multiple file mode
    THROTTLE=r	Throttle to max Mby/sec
    MAXLEN=n	Stop at this maximum file length
    NFFT=n	FFT size for PSD auxiliary file
    PSDR=n	Output PSD frame at n frames per second
    NAVG=n	FFTs to average for each output frame
    NCPA=n	Number of channels per address for network streams
    TOP=t	Time of interest in a multifile archive
    DUR=r	Duration of interest in a multifile archive
    EVENT=s	Event name of interest in a multifile archive
    GPS=s	Filename with updating GPS data
    RG=IFS	Resource group IceFileSystem for direct IO from mapped buffers
    
  @author Jeff Schoen
  @version %I%, %G%
*/
public class Archiver extends DataFile {

  public final static String propertyList = "Func,Multi,MaxLines,MaxLineSize,Channel,Channels,Cleanup";
  public final static String functionList = "Normal,ICEUnpack,SDDSUnpack,IceDemux,RawMulti,ICEMulti,SDDSMulti,VRTMulti";
  public final static int NORMAL=0, ICEUNPACK=1, SDDSUNPACK=2, ICEDEMUX=3, RAWMULTI=4, ICEMULTI=5, SDDSMULTI=6, VRTMULTI=7;
  public final static int NIO=4; // all modes >= this use Network IO

  public final static int SDDSBUFSZ=1088*1024, ICEBUFSZ=1024*64, MAXNIOBUF=8192+64, RAWMULTISZ=131072;

  private sourcepic sp;
  private NetIO nio;
  private ToC toc;
  private PSD psd;
  private KeyVector channels;
  private Channel channel;
  private int multi=0,nchan=0,func=0,sock=0;
  private double maxLineSize=Constants.B1G;
  private int maxLines=0;
  private DataFile dfSuper;
  private String aaux,aaux1,aaux2;
  private Parser auxes;
  private int naux=0;
  private int headerLength=0;
  private boolean netio;
  private long niobufp,niobuft;
  private SDDSPacket spkh;
  private VRTPacket vpkh;
  private ICEPacket ipkh;
  private Time stime;
  private int seqNext=0;
  private int seqNextChn[];
  private double niodelayfactor=0;
  private long maxlen=0;
  private int pktlen=0;
  private int pktflg=0;
  private int pktelem;
  private int nfft=8192;
  private int navg=16;
  private double psdr=1.0;
  private int ncpa=1;
  private String event="";
  private int tcmode;
  private boolean needTimeUpdate,needCloseOut,needCleanUp;
  private int ctxi,ctxm;
  private int nioFlg=0;
  private int tocr=1;
  private int eventCnt=0;
  private String gpsfn;

  public Archiver (sourcepic sp) {
    this.sp = sp;   // to get specs
    dfSuper = this; // for reference in Channel class
    tcmode = sp.getTCMode();
  }
  public Archiver () {
    dfSuper = this; // for reference in Channel class
  }

  public int getMulti() { return multi; }
  public String getFunc() { return Parser.get(functionList,func,-1); }
  public int getMaxLines() { return maxLines; }
  public double getMaxLineSize() { return maxLineSize; }
  public String getEvent () { return event; }

  public void setMulti (int channels) { multi = channels; }
  public void setNChan (int channels) { nchan = channels; }
  public void setFunc (String function) { func = Parser.find(functionList,function,func,-1); }
  public void setMaxLines (int lines) { maxLines = lines; }
  public void setMaxLineSize (double bytes) { maxLineSize = bytes; }
  public void setEvent (String str) { event = str; needCloseOut=true; }
  public void setCleanup (String str) { needCleanUp=true; }

  public void setThrottle (double rate) {
    double drate = (1.0/getDelta()) * getBPE();
    if (multi>0) drate *= multi;
    if (nchan>0) drate *= nchan;
    drate *= 1e-6; // in MBy/s
    if (rate>0) drate = rate;
    else drate *= -rate;
    niodelayfactor = (rate==0 || drate==0)? 0 : 1.0/drate;
  }

  public boolean open() {

    // is this a network output
    String fname = fn.getFullName();
    netio = fname.startsWith("udp:") || fname.startsWith("stp:");
    String tfname = netio? "network_toc" : fn.getRoot()+"_toc";
    String pfname = netio? "network_psd" : fn.getRoot()+"_psd";

    if (netio) flags &= ~NATIVE;
    isInput  = (flags&INPUT)!=0;
    isOutput = (flags&OUTPUT)!=0;

    Object qual;
    qual = this.getQualifier("FUNC"); if (qual!=null) setFunc((String)qual);
    qual = this.getQualifier("MULTI"); if (qual!=null) setMulti(Convert.o2l(qual));
    qual = this.getQualifier("THROTTLE"); if (qual!=null) setThrottle(Convert.o2d(qual));
    qual = this.getQualifier("MAXLEN"); if (qual!=null) maxlen = Convert.o2x(qual);
    qual = this.getQualifier("PKTLEN"); if (qual!=null) pktlen = Convert.o2l(qual);
    qual = this.getQualifier("PKTFLG"); if (qual!=null) pktflg = Convert.o2l(qual);
    qual = this.getQualifier("NFFT"); if (qual!=null) nfft = Convert.o2l(qual);
    qual = this.getQualifier("NAVG"); if (qual!=null) navg = Convert.o2l(qual);
    qual = this.getQualifier("PSDR"); if (qual!=null) psdr = Convert.o2d(qual);
    qual = this.getQualifier("NCPA"); if (qual!=null) ncpa = Convert.o2l(qual);
    qual = this.getQualifier("NIOTHR"); if (qual!=null) nioFlg = NetIO.THREAD;
    qual = this.getQualifier("TOCR"); if (qual!=null) tocr = Convert.o2l(qual);
    qual = this.getQualifier("GPS"); if (qual!=null) gpsfn = (String)qual; 

    // open the table of contents
    toc = new ToC (this,tfname,maxLines,maxLineSize,flags);
    if (isOutput && psdr>0)  psd = new PSD (this,pfname,nfft,flags);

    // add the GPS info
    if (isOutput && gpsfn!=null && gpsfn.length()>0  && !gpsfn.equalsIgnoreCase("NONE") && toc!=null) {
      Table gpst = null;
      TextFile tf = new TextFile(gpsfn);
      if (tf.exists()) gpst = new Table(tf);
      if (gpst!=null) toc.df.keywords.put("GPS",gpst.toString());
      else { M.warning("Could not open GPS file="+gpsfn+" for archive="+getURL()); gpsfn = null; }
    } else gpsfn=null;

    qual = this.getQualifier("TOP"); if (qual!=null) setTop(Convert.o2d(qual));
    qual = this.getQualifier("DUR"); if (qual!=null) setDur(Convert.o2d(qual));
    qual = this.getQualifier("EVENT"); if (qual!=null) setEvent(Convert.o2s(qual));

    ipkh = (ICEPacket)pkh;
    if (func>=NIO) {		// network IO open
      if (pkh==null) M.error("Must use /APACKET=ICE for NetIO options");
      nio = new NetIO(true);
      sock = nio.nioOpen(fn.getFullName(),multi,nioFlg);
      niobufp = nio.nioAlloc(MAXNIOBUF);
      if (pktlen==0) pktlen = ipkh.getSize()*ipkh.getBPA();
      pktelem = pktlen/ipkh.getBPA();
      if (func==ICEMULTI) ipkh = new ICEPacket( getFormat(), pktelem, pktflg);
      if (func==SDDSMULTI) spkh = new SDDSPacket( getFormat(), SDDSPacket.ICE );
      if (func==VRTMULTI) vpkh = new VRTPacket( getFormat(), pktelem, pktflg );
      if (func==SDDSMULTI) spkh.setDelta( getDelta() );
      if (func==VRTMULTI) { int rate=(int)(1/getXDelta()+.5); vpkh.createContextFor(1,rate,0,0); ctxm = rate/pktelem; }
      stime = new Time();
      isOpen = (sock>0);
      headerLength = (func==VRTMULTI)? vpkh.getHeaderLength() : 64;
      seqNext = 0;
    }
    else if (func==RAWMULTI) {
      super.open();
      offset = 0;
      String curl = this.getURL().replace(".tmp",  "");
      channels = new KeyVector(nchan);
      Channel c;
      for (int m=0; m<nchan; m++) {
        c = new Channel(this, curl, m);
        c.open(m);
        channels.put(m, "CHAN_"+m, c);
      }
      isOpen = true;
    }
    else if (func==ICEUNPACK) {
      nio = new NetIO(true);
      sock = nio.nioOpen(fn.getFullName(),1,0);
      isOpen = (sock>0);
    }
    else if (multi<=0) {	// single channel
      channel = new Channel(this,multi);
      channel.tocr.setFreq(getChannelFreq());
      channel.tocr.setTime(getChannelTime().getSec());
      channel.open(-1);
      if (multi==0) super.open();	// this is the actual archive file
      else isOpen = channel.df.isOpen;
      needTimeUpdate = (tcmode>0);
    }
    else {			// multi mode file segments are opened on demand
      channels = new KeyVector(multi);
      for (int i=0; i<multi; i++) {
        Channel c = new Channel(this,i);
        channels.put(i,"CHAN_"+i,c);
      }
      super.open();		// this is now a file collection that access all the others
    }
    int maxChan = Math.max(1,Math.max(multi,nchan));
    seqNextChn = new int[maxChan];
    if (ipkh!=null) ipkh.setTC(tcmode,0,0.0,0.0,0.0);
    needCloseOut=false;
    return isOpen;
  }

  public void flush() {
    if (!isOutput || !isOpen) return;
    if (nio!=null) {
      // netio always flushed
    } else {
      if (func==RAWMULTI);
      else if (multi<=0) channel.flush();
      else for (int i=0; i<multi; i++) getChannel(i).flush();
      if (multi==0) super.flush();
    }
    if (toc!=null) toc.flush();
    if (psd!=null) psd.flush();
  }

  public void close() {
    if (nio!=null) {
      nio.nioClose(sock);
      nio.nioFree(niobufp);
    } else {
      if (func==RAWMULTI) for (int i=0; i<nchan; i++) getChannel(i).close();
      else if (multi<=0) channel.close();
      else for (int i=0; i<multi; i++) getChannel(i).close();
      if (multi==0) super.close();
    }
    if (toc!=null) { toc.close(); if (needCleanUp) toc.df.erase(true); }
    if (psd!=null) { psd.close(); if (needCleanUp) psd.df.erase(true); }
  }

  public void setPktHeader (ICEPacket icepkh) {
    if (func!=ICEMULTI) ipkh.copyFrom(icepkh);
    else ((ICEPacket)pkh).copyFrom(icepkh);
  }

  public int read (long lbuf, byte[] buf, int boff, int bytes, int chan) {
    int stat=0; 
      if (lbuf!=0) stat = read(lbuf,boff,bytes);
      else if (buf!=null) stat = read(buf,boff,bytes);
      channel.bytes += stat;
    return stat;
  }

  public int write (long lbuf, int boff, int bytes, int chan) {
    return write (lbuf, null, boff, bytes, chan);
  }

  public int write (long lbuf, byte[] buf, int boff, int bytes, int chan) {
    int stat=0; 
    if (nio!=null) {
      int niodelay = (int)(niodelayfactor*bytes);
      int ic = chan;
      if (ic<0) ic=0;
      if (ncpa>1) ic/=ncpa;	/* adjust for channels per address */
      // compute header
      if (func==ICEMULTI) {
        ICEPacket jpkh = (ICEPacket)pkh;
	ipkh.setCount(seqNextChn[ic]);
	ipkh.setChannel(chan+1);
	ipkh.setTC(jpkh); 
        double delta = jpkh.getUserDataD(1);
        if (delta>0) ipkh.setUserData(delta,1);
	if (chan==-2) {
          Native.ja2p(ipkh.buf,0, niobufp,0, headerLength);
	  nio.nioSendBank(sock, nchan,ncpa, niobufp,headerLength,pktlen, lbuf+boff,bytes, 0);
	  ipkh.upCount(bytes/pktlen/(nchan/ncpa));
	}
	else for (int i=0; i<bytes; i+=pktlen) {
	  if (buf!=null) {
            Native.ja2p(ipkh.buf,0, niobufp,0, headerLength);
            Native.ja2p(buf,boff+i, niobufp,headerLength, pktlen);
            nio.nioSend(sock,ic,niobufp,headerLength+pktlen,(i==0)?niodelay:0);
          }
	  else if (boff>=headerLength) { // OK to overwrite old data buffer with packet header for speed
	    Native.ja2p(ipkh.buf,0, lbuf+boff+i,-headerLength, headerLength);
	    nio.nioSend(sock,ic,lbuf+boff+i-headerLength,headerLength+pktlen,(i==0)?niodelay:0);
	  } else {
            Native.ja2p(ipkh.buf,0, niobufp,0, headerLength);
            Native.p2p(lbuf,boff+i, niobufp,headerLength, pktlen);
            nio.nioSend(sock,ic,niobufp,headerLength+pktlen,(i==0)?niodelay:0);
	  }
	  ipkh.upCount();
	  if (bytes!=pktlen) ipkh.decTCO(pktelem);
	}
	seqNextChn[ic]=ipkh.getCount();
      }
      else if (func==SDDSMULTI) {
	double delta = ipkh.getUserDataD(1);	// abscissa delta set ?
	if (delta>0) spkh.setDelta(delta);	// possible variable delta
	else delta = getDelta();		// fixed delta already set
	ipkh.getTC(stime,0.0,delta); 		// get TC of first sample of ICE packet
	if (ipkh.getTCStatus()>0)
	spkh.setTC(0,delta,stime.getWSec(),stime.getFSec()); // set TC of first SDDS packet
	spkh.setCount(seqNextChn[ic]);
	niodelay = (int)(niodelayfactor*1024);
	for (int i=0; i<bytes; i+=1024) {
          Native.ja2p(spkh.buf,0, niobufp,0, headerLength);
	  if (buf!=null) {
            Native.ja2p(buf,boff+i, niobufp,headerLength, 1024);
	  } else {
            Native.p2p(lbuf,boff+i, niobufp,headerLength, 1024);
	  }
          nio.nioSend(sock,ic,niobufp+8,1080, niodelay);
	  if (i==0) spkh.setTC(0,0.0,0.0,0.0);	// no valid SDDS TC on the rest
	  spkh.upCount();
        }
	seqNextChn[ic]=spkh.getCount();
      } 
      else if (func==VRTMULTI) {
	for (int i=0; i<bytes; i+=pktlen) {
	  if (ctxi==0 && vpkh.cbuf!=null) {	// context packet
	    int lc = vpkh.ctx2pkt(niobufp);
            nio.nioSend(sock,ic,niobufp, lc, 0);
	  }
          Native.ja2p(vpkh.buf,0, niobufp,0, headerLength-4);	// 4 byte trailer
	  if (buf!=null) Native.ja2p(buf,boff+i, niobufp,headerLength-4, pktlen);
	  else           Native.p2p(lbuf,boff+i, niobufp,headerLength-4, pktlen);
          Native.ja2p(vpkh.buf,headerLength-4, niobufp,headerLength+pktlen-4, 4);
          nio.nioSend(sock,ic,niobufp,headerLength+pktlen, niodelay);
	  vpkh.upCount();
	  if (++ctxi>=ctxm) ctxi=0;
        }
      } 
      else if (func==ICEUNPACK) {
	for (int i=0; i<bytes; i+=pktlen) {
	  nio.nioSend(sock,ic,lbuf+i,pktlen,(i==0)?niodelay:0);
	}
      }
    }
    else if (func==SDDSUNPACK && chan<0) {
      int ndo=Math.min(bytes,1088), bend=boff+bytes;
      for (; boff<bend; boff+=ndo) {
	chan = findSDDSChannel (lbuf,boff,bytes);
	if (chan<0) M.error("Out of archive channels");
	stat += write (lbuf,boff,ndo,chan);
      }
    }
    else if (func==ICEDEMUX && chan<0) {
      int ndo=1024, bend=boff+bytes;
      for (; boff<bend; boff+=ndo) {
	chan = findDemuxChannel (lbuf,boff,bytes);
	stat += write (lbuf,boff,ndo,chan);
      }
    }
    else if (func==RAWMULTI) {
      int cbytes = (int) (( (maxlen<=0) || (offset+bytes <= maxlen) ) ? bytes : maxlen-offset);
      if ( cbytes > 0 ) {
        Channel c;
        int step = getFrameSize()*bpa;      
        for ( int m=0; m<cbytes; m+=step ) {
          c = getChannel(seqNext);
          Native.p2p(lbuf, m, c.lbuf, c.inbuf, step);
          c.inbuf += step;   
          if ( c.inbuf+step > RAWMULTISZ ) {
            // next pass will overflow, write data
            c.df.write(c.lbuf, 0, c.inbuf);
            c.inbuf = 0;
          }
          // increment channel number (safely)
          seqNext = (seqNext+1)%nchan;
        }
        offset += cbytes;
        stat = cbytes;
      }
    }
    else if (multi==0) {
      if (lbuf!=0) stat = write(lbuf,boff,bytes);
      else if (buf!=null) stat = write(buf,boff,bytes);
      if (psd!=null) psd.proc(lbuf,buf,boff,bytes);
      channel.bytes += stat;
    }
    else {
      if (chan<0 && multi==1) chan=0;
      Channel c = getChannel(chan);
      if (needCloseOut) {
	c.close();
        needCloseOut=false;
      }
      if (!c.isOpen) {
        Time t = getChannelTime();
	needTimeUpdate = (tcmode>0);
        int tocoff = (multi>1)? chan : -1;
        if (func==SDDSUNPACK) t = SDDSPacket.getTC(null,boff,lbuf);
	c.tocr.setFreq(getChannelFreq());
	c.tocr.setTime(t);
	c.tocr.setEvent(event);
	c.time.fromTime(t);
	c.open(tocoff);
	if (func==SDDSUNPACK) c.df.keywords.putMain("PACKET","SDDS/ICE");
      }
      if (needTimeUpdate) {
       ICEPacket jpkh = (ipkh!=null)? ipkh : (sp!=null)? sp.getAPkt() : null;
       if (jpkh!=null && jpkh.getTCStatus()>0) {
	Time t = c.time; double tcoff = c.df.getOffset();
	jpkh.getTC(t,-tcoff,getXDelta()); // more accurate time from packet header
	c.tocr.setTime(t);
        if (toc!=null) toc.df.write (c.tocr.dtoc,(double)toc.offset,1);
	setTimeNicely(c.df,t);
	needTimeUpdate = false;
       }
      }
      if (c.buf!=null) {	// cache small writes
	Native.p2ja(lbuf,boff, c.buf,c.inbuf, bytes); c.inbuf += bytes;
	if (c.inbuf>=c.buf.length) { c.df.write(c.buf,0,c.inbuf); c.inbuf=0; }
	stat = bytes;
      } else {
        if (lbuf!=0) stat = c.df.write(lbuf,boff,bytes);
        else if (buf!=null) stat = c.df.write(buf,boff,bytes);
      }
      if (psd!=null) psd.proc(lbuf,buf,boff,bytes);
      c.bytes += stat;
      if (maxLineSize>0 && c.bytes >= maxLineSize) c.close();
      c.hasNewData = true;
    }
    return stat;
  }

  public void addEventKey (String key, double value) {
    if (!isOutput || sp==null || toc==null || toc.df==null || !toc.df.isOpen) return;
    if (eventCnt==0 && seek()==0) return;
    eventCnt++;
    toc.df.keywords.put("EVENTS",eventCnt);
    toc.df.keywords.add("EVENT"+eventCnt,"{TIME="+sp.getTime().getSec()+","+key+"="+value+"}");
  }

  public static Archiver openForExport (Midas M, String path, byte[] hdr, int dir) {
    Archiver ha = new Archiver();
    int flags = (dir>0)? ha.INOUT|ha.APPEND : ha.INPUT;
    boolean validHeader = new String(hdr,0,4).equals("BLUE");
    if (dir<=0 || !validHeader) ha.init(M,path);
    else ha.init(M,path,hdr,flags);
    ha.open(flags|ha.NOABORT);
    ha.setMaxLines(ha.toc.nsf);
    ha.setMaxLineSize(ha.toc.lineSize);
    if (dir>0 && validHeader) {
      double dstart = Convert.unpackD(hdr,32);
      double dsize  = Convert.unpackD(hdr,40);
      long tbytes = (long)(dstart+dsize);
      ha.toc.topLength = (long)dsize;
    }
    return ha;
  }

  public long getExportSize() {
    DataFile df = channel.df;
    long bytes = 512;
    if (toc.topLength>0) bytes += toc.topLength;
    else bytes += getLengthBytes();
    return bytes;
  }

  public int readExport (byte[] buf, long offset, int bytes) {
    if (offset==0) {
      DataFile df = channel.df;
      DataFile dfo = new DataFile(M,"NULL","1000",df.getFormat(),DataFile.OUTPUT);
      dfo.setXUnits(df.getXUnits());
      dfo.setXDelta(df.getXDelta());
      dfo.setDataSize(getExportSize()-512);
      Time tc = df.getTime();
      if (toc.topIndex!=0) tc.addSec( df.getXDelta()*toc.topIndex/getBPE() );
      dfo.setTime(tc);
      System.arraycopy(dfo.hb,0, buf,0, 512);
      return 512;
    } else {
      long doff = toc.readOffset(offset-512);
      if (toc.isf != channel.offset) channel.open(toc.isf); 
      channel.df.io.seek(doff);
      channel.df.io.read(buf,0,bytes);
    }
    return bytes;
  }

  public int writeExport (byte[] buf, int bytes) {
    return write (0L, buf, 0, bytes, 0);
  }

  public ToC getToC() { return toc; }
  public PSD getPSD() { return psd; }

  public double getLengthBytes() {
    double len;
    if (multi<0) {
      len = toc.size*toc.lineSize;
    } else {
      len = getDataSize();
    }
    return len;
  }

  public double getLength() {
    double len=getLengthBytes()/getBPE();
    return len*getDelta();
  }

  public double getOffset() {
    double off;
    if (multi<0) {
      off = channel.offset*toc.lineSize/getBPE() + channel.df.getOffset();
    } else {
      off =  super.getOffset();
    }
    return off;
  }

  public void setAAux (String aux) { 
    naux = 0;
    if (aux==null || aux.length()==0) return;
    int ic = aux.indexOf("<>");
    if (ic>0) {  // parse i1:i2 syntax
      int i1 = Convert.s2l(aux.substring(0,ic));
      int i2 = Convert.s2l(aux.substring(ic+2));
      aux = ""+i1; for (int i=i1+1; i<=i2; i++) aux += "|"+i;
    }
    if (aux.indexOf('|')>0) { 
      auxes = new Parser(aux,'|');
      naux = auxes.elements();
    }
    else {
      naux=1; 
      aaux = aux;
    }
  }
  public void setAAux1 (String aux) { 
    if (aux==null || aux.length()==0) return;
    M.deprecate("AAUX1=x,AAUX2=y changed to AAUX=x|y");
    aaux1 = aux; if (aaux2!=null) setAAux (aaux1+"|"+aaux2);
  }
  public void setAAux2 (String aux) { 
    if (aux==null || aux.length()==0) return;
    M.deprecate("AAUX1=x,AAUX2=y changed to AAUX=x|y");
    aaux2 = aux; if (aaux1!=null) setAAux (aaux1+"|"+aaux2);
  }

  public double getChannelFreq() {
    if (sp!=null) return sp.getRfFreq()*1e6 + sp.getFreq();
    else return 0.0;
  }
  public Time getChannelTime() {
    if (sp!=null) return sp.getTime();
    else return getTime();
  }
  public String getChannelFormat() {
    if (sp!=null) return sp.getFormat();
    else return getFormat();
  }

  public Channel getChannel() { return channel; }
  public Channel getChannel (int i) { return (channels!=null)? (Channel)channels.get(i) : channel; }
  public KeyVector getChannels() { return channels; }

  byte[] buffer = new byte[8];

  public int findSDDSChannel (long lbuf, int boff, int bytes) {
    Native.p2ja(lbuf,boff,buffer,0,8);
    Convert.swap4(buffer,4,1);
    int addr = Convert.unpackL(buffer,4);
    Convert.swap2(buffer,2,1);
    short port = Convert.unpackI(buffer,2);
    int iopen = -1;
    for (int i=0; i<multi; i++) {
      Channel c = (Channel)channels.get(i);
      if (c.port==port && c.addr==addr) return i;
      else if (c.isOpen);
      else if (iopen<0) iopen=i;
    }
    if (iopen>=0) {
      Channel c = (Channel)channels.get(iopen);
      c.port=port; c.tocr.setPort(port);
      c.addr=addr; c.tocr.setAddress(addr);
    }
    return iopen;
  }

  public int findDemuxChannel (long lbuf, int boff, int bytes) {
    Native.p2ja(lbuf,boff,buffer,0,8);
    int chan = (buffer[0]&0x1) + ((buffer[2]&0x1)<<1) + ((buffer[4]&0x1)<<2) + ((buffer[6]&0x1)<<3);
    return chan;
  }

  public void checkChannels() {
    for (int i=0; i<multi; i++) {
      Channel c = (Channel)channels.get(i);
      if (c.isOpen && !c.hasNewData) {
	//System.out.println("Check Closing "+c+" "+Time.tag());
        c.close();
      }
      c.hasNewData = false;
    }
  }

  public void discontinue() {
    if (multi<0) {
      Channel c = getChannel(0);
      c.close();
      // dont leave stale packet in buffer
      if (ipkh!=null) ipkh.setTC(tcmode,0,0.0,0.0,0.0);
    }
  }

  public boolean isFinished() {
    return ( isOpen && maxlen>0 && offset>=maxlen );
  }

  public void setTop (double top) {
    if (toc!=null) toc.setTop(top);
  }
  public void setDur (double dur) {
    if (toc!=null) toc.setDur(dur);
  }
  public void setTime (Time time) {
    setTimeNicely(this,time.getWSec(),time.getFSec());
  }
  public void setTimeNicely (DataFile df, Time time) {
    setTimeNicely(df,time.getWSec(),time.getFSec());
  }
  public void setTime (double wsec, double fsec) {
    setTimeNicely(this,wsec,fsec);
  }
  public static void setTimeNicely (DataFile df, double wsec, double fsec) {
    double timeu = 1e-6  * Math.floor(fsec*1e6+.5);
    int timep = (int)Math.floor((fsec-timeu)*1e12+.5);
    df.setTimeCode(wsec+timeu);
    df.keywords.putMain("TC_PREC",""+timep+"e-12");
  }

  // Handles a data Channel for multifile archives
  public class Channel {
    short port;
    int addr;
    int offset;
    double bytes;
    boolean isOpen;
    boolean hasNewData;
    byte[] buf;
    int inbuf=0;
    long lbuf;
    DataFile df;
    Time time = new Time();
    FileName mfn;
    ToCR tocr;
    String quals;

    public Channel (DataFile df, int index) {
      this.df = df;
      tocr = new ToCR(toc.df);
      tocr.setEvent(getEvent());
      tocr.setStatus("");
      tocr.setKey((short)(index+1));
      tocr.setPort((short)0);
      tocr.setAddress(0);
      tocr.setFormat(getChannelFormat());
      tocr.setFreq(0.0);
      tocr.setTime(0.0);
      if (func==SDDSUNPACK) buf = new byte[SDDSBUFSZ];
      quals = fn.getQualifiers();
      if (quals.indexOf("+APPEND")>0) quals = quals.replace("+APPEND","|");
      else if (quals.indexOf("APPEND")>0) quals = quals.replace("APPEND","|");
    }
    public Channel (DataFile dfi, String url, int index) {
      this(dfi, index);
      String cname = String.format("%s_ch_%04d", url, index);
      mfn = new FileName(cname);
      mfn.setExt("tmp");
      lbuf = FileIO.malloc(RAWMULTISZ);
      inbuf = 0;
      buf = null;
    }
    public boolean open (int offset) {
      if (isOpen) close();
      if (offset<0) offset = toc.offset;
      tocr.setStatus("OPENED");
      bytes = 0;
      boolean stat=true;
      if (multi!=0) {
	int suffix = offset+1;
	tocr.setKey((short)suffix);
	String format = fn.getRoot()+"_%d";
        String fname = String.format(format, suffix);
	Object ref=cmd; if (ref==null) ref=M;
	if (isInput) df = new DataFile(ref,fname+quals,"1000","",flags&~APPEND);
	if (isOutput) df = new DataFile(ref,fname+quals,dfSuper,flags&~APPEND);
	int entry = offset;
        if (naux==1) df.setAux(aaux);
        else if (naux>1) df.setAux(auxes.get((entry%naux)+1));
	if (time!=null) setTimeNicely(df,time);
	stat = df.open();
      }
      else if (func==RAWMULTI) {
        Object ref=cmd; if (ref==null) ref=M;
        df = new DataFile(ref, mfn, "1000", getFormat(), 0);
        df.setDetached(1);
        stat=df.open(DataFile.OUTPUT|DataFile.NATIVE);
        df.setDataStart(0);
        df.setXUnits(Units.TIME_S);
        df.setXDelta(dfSuper.getXDelta()*nchan);
      }
      if (dfSuper.isOutput) {
        tocr.setRate(1.0/df.getXDelta());
        tocr.setLength(0.0);
        toc.write(tocr,offset);
      }
      this.offset = offset;
      isOpen = stat;
      return stat;
    }
    public void flush() {
      if (!dfSuper.isOutput || !isOpen) return;
      if (inbuf>0) { 
        if (func==RAWMULTI) df.write(lbuf, 0, inbuf, false);
        else df.write(buf,0,inbuf);
        inbuf=0;
      }
      if (multi!=0) df.flush();
      double databytes = bytes;
      if (func==SDDSUNPACK) databytes = bytes*1024/1088;
      tocr.setLength( (databytes/df.dbpe)*df.getDelta() );
    }
    public void close() {
      if (!isOpen) return;
      tocr.setStatus ("CLOSED");
      tocr.setFreq(getChannelFreq());
      updateKeys(df);
      flush();
      if (multi!=0 && df!=dfSuper) df.close();
      else if (func==RAWMULTI) {
        df.close(); 
        FileIO.mfree(lbuf);
      }
      if (dfSuper.isOutput) {
	if (offset<toc.size) { 	// check for external modifications to fields
	  toc.read(offset);
	  tocr.setEvent(toc.tocr.getEvent());	// copy event changes
	  if (toc.tocr.getStatus().equals("OPENED/S")) tocr.setStatus ("SAVED");
	}
	toc.write (tocr,offset);
	if (multi!=0 && psd!=null) psd.write (offset);
      }
      if (multi<=0) toc.nextOffset();
      isOpen = false;
    }
    public void updateKeys (DataFile df) {
      if (!isOutput) return;
      if (sp==null) return;

      df.keywords.put("PORT",sp.getPort());
      if (sp.getRfFreq()>0) { 
	df.keywords.put("RFFREQ",sp.getRfFreq());
	df.keywords.put("RFGAIN",sp.getRfGain());
      }
      if (sp.isTuner()) { 
	df.keywords.put("FREQ",sp.getFreq());
	df.keywords.put("GAIN",sp.getGain());
      }
      Table akw = sp.getArchKW();
      if (akw != null) df.keywords.putAll(akw);
    }
  }

  public static int doFunc (Object ref, String func, String fname, double ttop, double tdur, String event, String file) {
    int status = 0;
    boolean doErase = func.equals("ERASE");
    boolean doAssign = func.equals("ASSIGN");
    boolean doSelect = func.equals("SELECT");
    boolean doImport = func.equals("IMPORT");
    boolean doExport = func.equals("EXPORT");
    boolean doProtect = func.equals("PROTECT");
    boolean doUnProtect = func.equals("UNPROTECT");
    boolean allEvents = (event==null) || (event.length()==0) || event.equals("*");
    int flags = DataFile.INPUT | DataFile.OUTPUT;
    ToC toc = new ToC(ref,fname+"_toc",-1,-1,flags);
    ToCR tocr = toc.tocr;
    for (int i=0; i<toc.size; i++) {
      toc.read(tocr,i); 
      int ckey = i+1;
      double ctop = tocr.getTime();
      double cdur = tocr.getLength();
      String cevn = tocr.getEvent();
      boolean opened = tocr.getStatus().startsWith("OPENED");
      boolean match = doAssign || allEvents || event.equals(cevn);
      if (opened) cdur = toc.lineSize/(Data.getBPA(tocr.getFormat())*tocr.getRate());
      if (ttop<0) match &= (i>=(-ttop-1)) && (i<(-ttop-1-tdur));	// by index
      else        match &= (ctop+cdur>=ttop) && (ctop<=(ttop+tdur));	// by time
      if (match) {	// event match and time overlapped
        if (doAssign) tocr.setEvent(event);
        if (doProtect) tocr.setStatus(opened?"OPENED/S":"SAVED");
        if (doUnProtect) tocr.setStatus("CLOSED");
	if (doAssign||doProtect||doUnProtect) toc.write(tocr,i);
        status++;
      }
    }
    toc.close();
    return status;
  }

  public static String getFileAt (Object ref, String fname, double ttop, double tdur, int xfer) {
    int flags = DataFile.INPUT;
    DataFile df = new DataFile(ref,fname+"_toc");
    if (!df.find(-1)) return fname;
    ToC toc = new ToC(ref,fname+"_toc",-1,-1,flags);
    if (ttop>0) toc.setTop(ttop);
    if (tdur>0) toc.setDur(tdur);
    int bpe = Data.getBPA(toc.tocr.getFormat());
    long top = toc.topIndex/bpe;
    long len = toc.topLength/bpe;
    top = Math.max(0,top/xfer)*xfer;	// round to xfer length
    String fn = fname+"_"+toc.tocr.getKey();
    if (ttop>0 && tdur>0) fn += "("+top+":+"+len+")";
    return fn;
  }

  // Handles a Table of Contents File
  public static class ToC {

    int offset,topOffset;
    DataFile df;
    ToCR tocr;
    int isf,nsf,size;
    boolean isInput,isOutput;
    long topIndex,topLength,clen;
    double lineSize,topTime;

    public ToC (Object ref, String fname, int nsf, double bpsf, int flags) {
      flags &= (INPUT|OUTPUT|APPEND);
      isInput = (flags&INPUT)!=0;
      isOutput = (flags&OUTPUT)!=0;
      df = new DataFile();
      df.init (ref,fname,"3000","NH",flags); // get same flags OUTPUT|APPEND as archive file
      df.setSubRecords("EVNT|1A,STAT|1A,KEY|SI,PORT|SI,ADDR|4B,FORM|1A,RATE|SD,FREQ|SD,TIME|SD,LEN|SD");
      if (isOutput&!isInput) {
	df.keywords.put("LINES",Convert.l2o(nsf));
	df.keywords.put("LINESIZE",Convert.d2o(bpsf));
	df.keywords.put("CUROFFSET",Convert.d2o(0.0));
      }
      df.open();
      this.nsf = df.keywords.getL("LINES",nsf);
      bpsf = df.keywords.getD("LINESIZE",bpsf);
      offset = df.keywords.getL("CUROFFSET",0);
      size = (int)df.getSize();
      lineSize = bpsf;
      tocr = new ToCR(df);
    }

    public void write (ToCR tocr, int offset) {
      df.write (tocr.dtoc,(double)offset,1);
      df.update();
      if (offset>=size) { size=offset+1; df.setSize(size); }
    }
    public void nextOffset (int offset) {
      this.offset = offset;
    }
    public int nextOffset() {
      offset++;
      if (nsf>0) {	// circular
        if (offset>=nsf) offset=0;
	while (offset<size) { // check for saved slice
	  read(tocr,offset);
	  if (tocr.getStatus().startsWith("SAVE")) offset++;
	  else break;
          if (offset>=nsf) offset=0;
	}
      }
      df.keywords.put("CUROFFSET",Convert.l2o(offset));
      if (isOutput) df.flush();
      return offset;
    }
    public long readOffset (long offset) {
      long doff = offset+topIndex;
      long dper = (long)lineSize;
      int  ioff = (int)(doff/dper);
      long joff = doff - ioff*dper;
      int  noff = (topOffset+ioff)%Math.max(1,size);
      isf = noff;
      return joff;
    }
    public void read (int offset) { read (tocr,offset); }
    public void read (ToCR tocr, int offset) { df.read (tocr.dtoc,(double)offset,1); }

    public void flush () { if (isOutput && df.isOpen()) df.flush(); }
    public void close () { if (df.isOpen()) df.close(); }
    public boolean isOpen () { return df.isOpen(); }
    public int size () { return size; }

    public double getLength() {
      double len=0;
      for (int i=0; i<size; i++) {
        read(tocr,i);
        len += tocr.getLength();
      }
      return len;
    }

    public void setTop (double top) {
      int i=0;
      double t=0, d=1;
      for (i=0; i<size; i++) {
        read(i);
        t=tocr.getTime();
        d=tocr.getLength();
        if (t<=top && top<t+d) break;
      }
      if (i>=size) { 
	System.out.println("Top time="+top+" not found. Starting at 1st record."); 
	i=0; read(i); t=tocr.getTime(); top=t;
      }
      topOffset = i;
      topIndex = (long)((top-t) * tocr.getRate()) * Data.getBPA(tocr.getFormat());
      topIndex = Math.max(0,topIndex);	// round to 1M
      topTime = top;
      nextOffset(i);
      clen=0;
    }

    public void setDur (double dur) { 
      topLength = (long)(dur * tocr.getRate() * Data.getBPA(tocr.getFormat()));
      topLength = Math.max(1,topLength/0x100000)*0x100000;	// round to 1M
    }

    public double getSavedSpace() {
      double len=0;
      for (int i=0; i<size; i++) {
	read(tocr,i);
	if (tocr.getStatus().equals("SAVED")) len += lineSize;
      }
      return len;
    }

    public double getTotalSpace() {
      return lineSize * size;
    }

  }

  // Handles one record of a Table of Contents File
  public static class ToCR {
    Data dtoc;
    public ToCR (DataFile df) {
      dtoc = df.getDataBuffer(1);
    }
    public void setEvent (String event)   { dtoc.packS(0,8,event); }
    public void setStatus (String status) { dtoc.packS(8,8,status); }
    public void setKey (short key)        { dtoc.packI(16,key); }
    public void setPort (short port)      { dtoc.packI(18,port); }
    public void setAddress (int addr)     { dtoc.packL(20,addr); }
    public void setFormat (String format) { dtoc.packS(24,8,format); }
    public void setRate (double rate)     { dtoc.packD(32,rate); }
    public void setFreq (double freq)     { dtoc.packD(40,freq); }
    public void setTime (double time)     { dtoc.packD(48,time); }
    public void setLength (double sec)    { dtoc.packD(56,sec); }

    public void setTime (Time time) { if (time!=null) setTime(time.getSec()); }

    public String getEvent()   { return dtoc.unpackS(0,8); }
    public String getStatus()  { return dtoc.unpackS(8,8); }
    public short  getKey()     { return dtoc.unpackI(16); }
    public short  getPort()    { return dtoc.unpackI(18); }
    public int    getAddress() { return dtoc.unpackL(20); }
    public String getFormat()  { return dtoc.unpackS(24,8); }
    public double getRate()    { return dtoc.unpackD(32); }
    public double getFreq()    { return dtoc.unpackD(40); }
    public double getTime()    { return dtoc.unpackD(48); }
    public double getLength()  { return dtoc.unpackD(56); }
  }

  // Handles a PSD File
  public class PSD {

    DataFile df;
    int nfft,nout,nrot,pass,itot,iper,spa,bpe;
    String format;
    boolean cx,isOutput;
    byte dtype;
    Fft fft;
    float[] wind,fbuf,fout;
    byte[] bbuf;

    public PSD (DataFile ha, String fname, int nfft, int flags) {

      this.format = ha.getFormat();
      this.nfft = nfft;
      spa = ha.getSPA();
      dtype = ha.getFormatType();
      bpe = (int)ha.getBPE();
      cx = (spa==2);

      double factor;
      double delx = ha.getXDelta();
      double dely = delx*nfft;
      double delf = 1.0/(nfft*delx);
      iper = (int)(((1.0/delx)/navg)*bpe);

      int fftf = Fft.FORWARD;
      if (cx) { nout=nfft; fftf|=(Fft.COMPLEX); factor=1.0/nfft; nrot=nfft/2-1; }
      else    { nout=nfft/2; fftf|=(Fft.REAL|Fft.PACKED); factor=0.5/nfft; nrot=0; }
      fft = new Fft(nfft,fftf);

      wind = Window.get("HANN",nfft,factor);
      fbuf = new float[nout*2];
      fout = new float[nout*2];
      bbuf = new byte[nout*8];

      flags &= (INPUT|OUTPUT|APPEND); // get same flags OUTPUT|APPEND as archive file
      isOutput = (flags&OUTPUT)!=0;
      df = new DataFile(ha.M,fname,"2000","CF",flags);
      df.setSubsize(nout);
      df.setXStart(-nrot*delf);
      df.setXDelta(delf);
      df.setXUnits(Units.FREQUENCY);
      df.setYUnits(Units.TIME);
      df.setYStart(0.0);
      df.setYDelta(1.0/psdr);
      df.open();
    }

    public void proc (long lbuf, byte[] buf, int boff, int bytes) {
      itot+=bytes; if (itot<iper) return; itot=0;
      int elem = bytes/bpe;
      if (elem>nfft) elem=nfft;
      if (lbuf!=0) {
	buf=bbuf; boff=0;
	int xfer = Math.min(bytes,nout*8);
	Native.p2ja(lbuf,0,buf,boff,xfer);
      } 
      Convert.bb2ja (buf,boff,dtype, fbuf,0,DataTypes.FLOAT, elem*spa);
      for (int i=elem*spa; i<nout*2; i++) fbuf[i]=0;
      if (cx) Multiply.CSC (fbuf,wind,fbuf,nfft);
      else    Multiply.SSS (fbuf,wind,fbuf,nfft);
      fft.work(fbuf);
      if (cx) fft.rotateCF(fbuf);
      if (pass==0) Noop.CC(fbuf,fout,nout);
      else Add.CCC(fbuf,fout,fout,nout);
      pass++;
      if (multi==0 && pass>=navg) write();
    }

    public void write (int index) {
      df.seek((double)index);
      write();
      df.update();
    }
    public void write () {
      Convert.ja2bb(fout,0,DataTypes.FLOAT, bbuf,0,DataTypes.FLOAT, nout*2);
      df.write (bbuf,0,nout*8);
      zero();
    }

    public void zero () { pass=0; itot=0; }
    public void flush () { if (isOutput) df.flush(); }
    public void close () { if (df.isOpen()) df.close(); }
    public boolean isOpen () { return df.isOpen(); }

  }

}
