/*
  Common defines for FPGA Core libraries in the ICE hardware environment.

*/

#ifndef v2print
#define v2print(...) if (p->verbose&0x2) fprintf(stderr,__VA_ARGS__)
#endif

#define DEBUG 0

/* Common Core register offsets */
#define COREG_SYS   0x00
#define COREG_DEC   0x04
#define COREG_GAIN  0x08
#define COREG_RATE  0x0C
#define COREG_RATIO 0x10
#define COREG_FRAME 0x14
#define COREG_FREQ  0x18
#define COREG_FLAG  0x1C
#define COREG_POFF  0x1C

#define COREG_COEF  0x20
#define COREG_CFIR  0x28
#define COREG_PFIR  0x30
#define COREG_RFIR  0x38

#define COREG_USER0 0x20
#define COREG_USER1 0x24
#define COREG_USER2 0x28
#define COREG_USER3 0x2C
#define COREG_USER4 0x30
#define COREG_USER5 0x34
#define COREG_USER6 0x38
#define COREG_USER7 0x3C

#define COREG_PLAN 0x100

#define COREG_SIG    0x7E0
#define COREG_PARAM0 0x7E0
#define COREG_PARAM1 0x7E4
#define COREG_PARAM2 0x7E8
#define COREG_PARAM3 0x7EC
#define COREG_PARAM4 0x7F0
#define COREG_PARAM5 0x7F4
#define COREG_PARAM6 0x7F8
#define COREG_PARAM7 0x7FC

/* Common Core sysreg bit register */
#define CORE_ENA  0x0001	/* enable core */
#define CORE_PLY  0x0002	/* output mode */
#define CORE_NOP  0x0004	/* noop mode */
#define CORE_XAB  0x0008	/* combined A/B mode */
#define CORE_DEC  0x0004
#define CORE_OPT  0x0008

#define CORE_TUNER 0x0010
#define CORE_OPT2  0x0020
#define CORE_OPT3  0x0040
#define CORE_THROTTLE 0x0080

#define CORE_PREP  0x400000
#define CORE_RESET 0x800000

#define CORE_I1B  0x0300
#define CORE_I4B  0x0200
#define CORE_I8B  0x0100
#define CORE_I16B 0x0000
#define CORE_I32B 0x0400
#define CORE_ICX  0x0800

#define CORE_O1B  0x3000
#define CORE_O4B  0x2000
#define CORE_O8B  0x1000
#define CORE_O16B 0x0000
#define CORE_O32B 0x4000
#define CORE_OCX  0x8000

/* Common helper macros */
#define I2L(A) ((int_4)A<<15)
#define IX2L(A) ((int_4)A<<16)
#define L2R(A) ((real_8)(A)*(2.147483648E9))
#define R2L(A) (int_4)rint((A)*(2.147483648E9))
#define R2LC24(A) r2lc((float)(A),24)
#define R2LC22(A) r2lc((float)(A),22)
#define R2LC20(A) r2lc((float)(A),20)
#define R2LC18(A) r2lc((float)(A),18)

int_4 r2lc (real_4 coef, int_4 bits) {
  int_4 lscl = (bits>=32)? 0x7FFFFFFF : (1<<(bits-1));
  int_4 l = (int_4)rint(coef*lscl);
  if (l>= lscl) l=lscl-1;
  if (l<=-lscl) l=1-lscl;
  l = l<<(32-bits);
  return l;
}

int_4 f2fptx (real_4 f) { return (int_4)(f*0x10000); }
real_4 fptx2f (int_4 x) { return ((real_4)x)/0x10000; }
int_8 d2dptx (real_8 d) { return (int_8)(d*B4G); }
real_8 dptx2d (int_8 x) { return ((real_8)x)/B4G; }

#ifndef CORE_IS_ICELIB
#include "icelib.h"
#endif

#ifndef CORE_VALUE_STRUCT
typedef struct {
  int_4 len;
  int_4 bpa;
  int_1 mode,type;
  int_1 bps,spa;
  int_1 *b;
} Value;
#define CORE_VALUE_STRUCT
#endif

#include "icehalo.h"

int_4 Core_copyHaloToSub (HALO *mhalo, HALO *shalo, int scid) {
  shalo->ph   = mhalo->ph  ;
  shalo->impl = mhalo->impl;
  shalo->flow = mhalo->flow;
  shalo->fmts = mhalo->fmts;
  shalo->flgs = mhalo->flgs;
  shalo->vbpr = mhalo->vbpr;
  shalo->mcid = mhalo->mcid;
  shalo->scid =        scid;
  return 0;
}

int_4 Core_config (void *handle, int_4 key, int_4 value) {
  HALO *pt = (HALO*)handle;
  if (key==0) pt->func = value;
  if (key==1) pt->fmts = value;
  if (key==2) pt->flow = value;
  if (key==3) pt->impl = value;
  if (key==4) pt->flgs = value;
  if (key==5) pt->scid = value;
  if (key==6) pt->mcid = value;
  if (key==7) pt->vbpr = value;
  if (key==6 && pt->impl==6 && pt->ph!=NULL) pt->ph->chan = value;
  return 0;
}

int_4 getEnables (void *handle) {
  HALO *halo = (HALO*)handle;
  int_4 sys = halo->flgs | CORE_ENA;
  return sys;
}

#define FMTS plan->halo.fmts
#define VBUFI(A,B) void *A=plan->halo.bi1; int_4 B=plan->halo.ni1
#define VBUFO(A,B) void *A=plan->halo.bo1; int_4 B=plan->halo.no1
#define VBUFI2(A,B) void *A=plan->halo.bi2; int_4 B=plan->halo.ni2
#define VBUFO2(A,B) void *A=plan->halo.bo2; int_4 B=plan->halo.no2
#define BBUFI(A,B) int_1 *A=plan->halo.bi1; int_4 B=plan->halo.ni1
#define BBUFO(A,B) int_1 *A=plan->halo.bo1; int_4 B=plan->halo.no1
#define IBUFI(A,B) int_2 *A=(int_2*)plan->halo.bi1; int_4 B=plan->halo.ni1<<1
#define IBUFO(A,B) int_2 *A=(int_2*)plan->halo.bo1; int_4 B=plan->halo.no1<<1
#define LBUFI(A,B) int_4 *A=(int_4*)plan->halo.bi1; int_4 B=plan->halo.ni1<<2
#define LBUFO(A,B) int_4 *A=(int_4*)plan->halo.bo1; int_4 B=plan->halo.no1<<2
#define FBUFI(A,B) real_4 *A=(real_4*)plan->halo.bi1; int_4 B=plan->halo.ni1<<2
#define FBUFO(A,B) ireal4 *A=(real_4*)plan->halo.bo1; int_4 B=plan->halo.no1<<2
#define keyMatch(A) isMatch(key,A)
#define keyMatchIndexed(A) isMatchIndexed(key,A)

int_4 isMatch (char *keyinp, char *keycfg) {
  int_4 ki = (keyinp[1]==':' && !(keycfg[1]==':' && keyinp[0]=='S'))? 2:0;
  int_4 kc = (keycfg[1]==':' && !(keyinp[1]==':' && keycfg[0]=='S'))? 2:0;
  int_4 lki=strlen(keyinp), lkc=strlen(keycfg); 
  if (lki>0 && keyinp[lki-1]=='?') lki--;
  if (lki-ki != lkc-kc) return 0;
  return strncmp(keyinp+ki,keycfg+kc,lki-ki)==0;
}

int_4 isSubMatch (char *a, char*b) {
  if (a[1]==':') a+=2;
  if (strstr(a,b)==a && a[strlen(b)]=='_') return 1;
  return 0;
}

char* subKey (char *a) {
  char* b = index(a,'_')+1;
  return b;
}

int_4 isMatchIndexed (char *keyinp, char *keycfg) {
  int_4 ki = (keyinp[1]==':')? 2:0;
  int_4 kc = (keycfg[1]==':')? 2:0;
  int_4 n  = strlen(keycfg)-kc;
  int_4 index = -1;
  if (strncmp(keyinp+ki,keycfg+kc,n)!=0) return index;
  if (keyinp[ki+n]<'0' || keyinp[ki+n]>'9') return index;
  index = atoi(keyinp+ki+n);
  if (ki==0 || kc==0 || keyinp[0]==keycfg[0]) return index;
  printf("KeyType=%c does not match configured Key=%s\n",keyinp[0],keycfg);
  return -1;
}

int_4 getFormatBits (int_4 fmts, int_4 n) {
  int_4 fmt = (fmts>>((n-1)*8)) & 0xFF;
  int_4 bps = 1;
  if ((fmt&0x8)!=0) { bps=2; fmt=fmt&0x7; } // complex
       if (fmt==0) bps *= 16; // int
  else if (fmt==1) bps *= 8;  // byte
  else if (fmt==2) bps *= 4;  // nibble
  else if (fmt==3) bps *= 1;  // bit
  else if (fmt==4) bps *= 32; // long
  else if (fmt==5) bps *= 64; // xlong
  else if (fmt==6) bps *= 32; // float
  else if (fmt==7) bps *= 64; // double
  return bps;
}

int_4 getFormatBytes (int_4 fmts, int_4 n) {
  int_4 byps = getFormatBits(fmts,n)>>3;
  return (byps<=0)? 1:byps;
}

int_4 getFormatMode (int_4 fmts, int_4 n) {
  int_4 fmt = (fmts>>((n-1)*8)) & 0xFF;
  int_4 spa = ((fmt&0x8)!=0)? 2 : 1;
  return spa;
}

int_4 getFmts (int_4 fmts) {
  int_4 reg = 0;
  reg |= (fmts&0x000F)<<8; // input buffer  to sysreg bit 8
  reg |= (fmts&0x0F00)<<4; // output buffer to sysreg bit 12
  return reg;
}

int_4 getFormat (const char *fmts) {
  char fm=fmts[0], ft=fmts[1];
  int_4 fmt = 0;
       if (ft=='I') fmt = 0x0;
  else if (ft=='B') fmt = 0x1;
  else if (ft=='N') fmt = 0x2;
  else if (ft=='P') fmt = 0x3;
  else if (ft=='L') fmt = 0x4;
  else if (ft=='X') fmt = 0x5;
  else if (ft=='F') fmt = 0x6;
  else if (ft=='D') fmt = 0x7;
  if      (fm=='C') fmt |= 0x8;
  return fmt;
}

/* Code for ICE Hardware mode */
#ifdef CORE_IS_ICE
#include "fpgalib.h"
#define HW_alloc(P,A) FPGA_open(P,A)
#define HW_free(P)    FPGA_close(P)
#define HW_read(A)    FPGA_sysrd(plan,A)
#define HW_write(A,B) FPGA_syswr(plan,A,B)
#define HW_wrblk(A,B,C) FPGA_syswrblk(plan,A,B,C)
#define HW_lock(A)    FPGA_lock(plan,A)

/* Code for Verilog Hardware Simulation mode */
#elif CORE_IS_VHS
#include "VCore.h"
#define HW_alloc(P,A) VCore_open(P,A)
#define HW_free(P)    VCore_close(P)
#define HW_read(A)    VCore_sysrd(plan,A)
#define HW_write(A,B) VCore_syswr(plan,A,B)
#define HW_wrblk(A,B,C) VCore_syswrblk(plan,A,B,C)
#define FPGA_wpb(P,A,B,C) VCore_syswr(P,B,C)
int_4 HW_lock (int_4 mode) { return 0; }

/* Code pass-thru for non-Hardware mode */
#else
int_4 HW_alloc (void *plan, void *context) { return 0; }
int_4 HW_free  (void *plan) { return 0; }
int_4 HW_read  (int_4 addr) { return 0; }
int_4 HW_write (int_4 addr, int_4 value) { return 0; }
int_4 HW_wrblk (int_4 addr, int_4 *value, int_4 size) { return 0; }
int_4 HW_lock (int_4 mode) { return 0; }
#endif

#define HW_INIT 0
#define HW_SET 1
#define HW_GET 2
#define HW_OPEN 3
#define HW_CLOSE 4
#define HW_PREP 5
#define HW_UPD 6

/*
  Core load methodology and address map:

  loadHdr @ level=0 initializes buffer for each core

  addr[1:0] - 0 = main block, 1 = route to next class, 2 = data element, 3 = array element
  addr[9:2] - variable number
  addr[23:10] - array variable index
  addr[31:24] - reserved for module select

  addr[9:2]=0xFF - sets the sub address - default to zero

  A5 - load Object - also sets Address to 0
  A4 - load Array
  A3 - load SubCore
  A2 - load Address
  A1 - load Variable
*/

#define MAXNEST 6
#define MAXLOADBUF 0x40000
#define ZBUFSIG 0x12345678

#define CORE_TYPE_LOCAL 4
#define CORE_TYPE_ICE 6

int_4 loadOff;
int_4 loadBuf[MAXLOADBUF];

int_4 HW_isLocal (void *plan) {
  HALO *halo = (HALO*)plan;
  int_4 local = (halo==NULL)? 0 : (halo->impl<=CORE_TYPE_LOCAL)? 1 : 0; 
  return local;
}
int_4 HW_needsPut (void *plan) {
  HALO *halo = (HALO*)plan;
  if (halo==NULL) return 0;
  if (halo->impl<=CORE_TYPE_LOCAL) return 0;
  return (halo->state==1)? 1:0;
}
void HW_loadAddr (void *plan, int_4 off) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA2000000|off;
}
void HW_loadHdr (void *plan, int_4 size) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA5000000|size;
}
void HW_loadSub (void *plan, int_4 index) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=0xA3000000|index;
}
void HW_loadInt8 (void *plan, int_8 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000002;
  loadBuf[loadOff++]=(value);
  loadBuf[loadOff++]=(value>>32);
}
void HW_loadInt4 (void *plan, int_4 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=value;
}
void HW_loadInt2 (void *plan, int_2 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=value;
}
void HW_loadInt1 (void *plan, int_1 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=value;
}
void HW_loadInt (void *plan, int_4 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=value;
}
void HW_loadFlt (void *plan, real_4 value) {
  if (HW_isLocal(plan)) return;
  loadBuf[loadOff++]=0xA1000001;
  loadBuf[loadOff++]=f2fptx(value);
}
void HW_loadDbl (void *plan, real_8 value) {
  if (HW_isLocal(plan)) return;
  HW_loadInt8(plan,d2dptx(value));
}
void HW_unloadInt8 (void *plan, int_4 index, int_8 *xval) {
  int_u4 val1,val2; int_4 off=0x100+(index*4); 
  if (HW_isLocal(plan)) return;
  val1 = HW_read(off);
  val2 = HW_read(off+4);
  *xval = (((int_8)val2)<<32)|val1;
}
void HW_unloadInt4 (void *plan, int_4 index, int_4 *lval) {
  if (HW_isLocal(plan)) return;
  int_4 val, off=0x100+(index*4); 
  val = HW_read(off);
  *lval = val;
}
void HW_unloadInt2 (void *plan, int_4 index, int_2 *ival) {
  if (HW_isLocal(plan)) return;
  int_4 val, off=0x100+(index*4); 
  val = HW_read(off);
  *ival = val;
}
void HW_unloadInt1 (void *plan, int_4 index, int_1 *bval) {
  if (HW_isLocal(plan)) return;
  int_4 val, off=0x100+(index*4); 
  val = HW_read(off);
  *bval = val;
}
void HW_unloadInt (void *plan, int_4 index, int_4 *lval) {
  if (HW_isLocal(plan)) return;
  int_4 val, off=0x100+(index*4); 
  val = HW_read(off);
  *lval = val;
}
void HW_unloadFlt (void *plan, int_4 index, real_4 *fval) {
  if (HW_isLocal(plan)) return;
  int_4 val, off=0x100+(index*4); 
  val = HW_read(off);
  *fval = fptx2f(val);
}
void HW_unloadDbl (void *plan, int_4 index, real_8 *dval) {
  int_8 xval;
  if (HW_isLocal(plan)) return;
  HW_unloadInt8 (plan,index,&xval);
  *dval = dptx2d(xval);
}

void HW_loadBuf (void *plan, void *value, char type, int_4 mode) {
  real_8 *dval=(real_8*) value;
  real_4 *fval=(real_4*) value;
  int_4  *lval=(int_4*)  value;
  int_8 xval;
  int i,sigx,size,xtra; 
  if (HW_isLocal(plan)) return;
  if (mode<=0) { goto NOLOAD; }
  /* signature and size of array are in an int_4 8 bytes before the array due to zallocBuf call */
  memcpy(&sigx,lval-2,4);
  if (sigx!=ZBUFSIG) { printf("Bad loadBuf memory sig=%08x. Skipping loadBuf.\n",sigx); goto NOLOAD; }
  memcpy(&size,lval-1,4); 
  if (size<0) { printf("Negative array size=%d. Skipping loadBuf.\n",size); goto NOLOAD; }
  if (size&0x3!=0) { printf("Array size=%d must be multiple of 4by. Skipping loadBuf.\n",size); goto NOLOAD; }
  if (size>=0x40000) { printf("Array too large. Only loading first 256 Kby of %d byte type %c array\n",size,type); size=0x40000; }
  size >>= 2;	/* to words */
  xtra = (type=='b')? 3 : (type=='s')? 1 : 0;
  if (loadOff+size>MAXLOADBUF) { printf("Too large loadBuf=%d for maxBuf=%d. Skipping loadBuf.\n",size,MAXLOADBUF); goto NOLOAD; }
  loadBuf[loadOff++]=0xA4000000|(xtra<<18)|(size&0x1FFFF);
       if (type=='f') for (i=0; i<size; i++) loadBuf[loadOff++]=f2fptx(fval[i]);
  else if (type=='d') for (i=0; i<size/2; i++) { xval=d2dptx(dval[i]); loadBuf[loadOff++]=xval; loadBuf[loadOff++]=xval>>32; }
  else                for (i=0; i<size; i++) loadBuf[loadOff++]=lval[i];
  return;
  NOLOAD:
  HW_loadInt(plan,0); 
  return;
}
void HW_loadObj (void *plan, int_4 size) {
  // handled by routine calculating size
}

#define SCID  0x5C
#define NCVAR 0x64

int_4 HW_push (void *plan, int_4 call) {
  HALO *halo = (HALO*)plan;
  int_4 i, i1=0, i2=0, j, status, val, valr, addr, cmd, size, xoff, level=-1, *lbuf=(int_4*)plan;
  int_4 loadObjSize[MAXNEST], loadObjOff[MAXNEST];
  int_4 isICE = (halo->impl==CORE_TYPE_ICE)? 1 : 0; 
  if (HW_isLocal(plan)) return 0;

  if (call==HW_PREP) {
    HW_lock(1);
    if (halo->scid>0) HW_write(SCID,halo->scid);
    loadOff=0;
  }
  else if (call==HW_OPEN) {
    halo->sys = getFmts(halo->fmts) | halo->flgs;
    /* write the system register initial reset */
    for (i=0; i<4; i++) HW_write(0,halo->sys|CORE_RESET);
    /* write the system register disabled */
    HW_write(0,halo->sys|CORE_PREP);
    level=-1; i1=0; i2=loadOff;
  }
  else if (call==HW_UPD) {
    level=0; i1=0; i2=loadOff;
    loadObjOff[level]=0;
    loadObjSize[level]=-1;
  }
  else if (call==HW_CLOSE) {
    halo->sys &= ~CORE_ENA;
    HW_write(0,halo->sys);
  }

  /* write the plan parameters (starting at 0x100) */
  for (i=i1; i<i2; ) {
    val = loadBuf[i++];
    cmd = val&0xFFF00000;
    size = val&0xFF;
    if (cmd == 0xA5000000) {				/* next object */
      addr = (level<0)? 0 : ((level==0)?0x100:0) + (loadObjOff[level]<<2);
      level++;
      loadObjOff[level]=0;
      loadObjSize[level]=size;
      if (level==0) HW_write(NCVAR,size);		/* num class vars for debug */
      addr |= ((1<<level)-1)<<10;
      HW_write(addr|1,0);
    }
    else if (cmd == 0xA2000000) {			/* set next address */
      loadObjOff[level] = (val&0xFF);
    }
    else if (cmd == 0xA1000000) {			/* set variable */
      for (j=0; j<size; j++) {
        val = loadBuf[i++];
        addr = ((level==0)?0x100:0) + (loadObjOff[level]<<2);
        HW_write(addr|2,val);
        if (level==0 && call==HW_OPEN) {		/* validate CORE Ram Block */
          valr = HW_read(addr);
          if (valr!=val) printf("Err reading plan level=%d register=%d wrote=%08x read=%08x\n",level,addr,val,valr);
        }
        loadObjOff[level]++;
      }
    }
    else if (cmd == 0xA4000000) {			/* set array object */
      addr = ((level==0)?0x100:0) + (loadObjOff[level]<<2);
      size = val&0x1FFFF;
      xoff = (val&0xC0000);
      HW_write((((1<<level)-1)<<10)|addr|1,xoff);	/* zero index register and set type */
      HW_wrblk(addr|3,loadBuf+i,size);			/* auto index increment */
      loadObjOff[level]++;
      i += size;
    }
    else {						/* other register */
      printf("HW push queue index=%d of %d cmd=%08x not recognized\n",i,i2,val);
    }
    while (level>=0 && loadObjOff[level]==loadObjSize[level]) {	/* back out of object */
      if (--level<0) break;
      loadObjOff[level]++;
      addr = ((1<<level)-1)<<10 | (loadObjOff[level]<<2);
      HW_write(addr|1,0);
    }
  }

  if (call==HW_OPEN) {
    val = (halo->flgs>>24)&0xFF;
    if (val!=0) {
      HW_write(0x0FFC,0x00);		/* write the trace register reset */
      HW_write(0x0FFC,val);		/* write the trace register set */
    }
    halo->sys |= (halo->vbpr<<20);	/* add verbose/debug flags */
    if (!isICE) halo->sys |= CORE_ENA;	/* default enable - ICE handled by core_enable calls */
    HW_write(0,halo->sys);		/* write the system register */
  }
  if (call==HW_OPEN || call==HW_UPD) { 
    if (halo->scid>0) HW_write(SCID,0);
    HW_lock(-1);
  }
  if (call==HW_OPEN) halo->state=1;
  if (call==HW_CLOSE) halo->state=0;
  
  return 0;
}

int_4 HW_dump (void *plan, int_4 call) {
  HALO *halo = (HALO*)plan;
  return 0;
}

/* Common FPGA implementation helpers (icelib.c) */
int_4 pow2sizem1 (int_4 size) {
  int_4 p2=0; for (;(1<<p2)<size;p2++);
  return (p2<<16)|(size-1);
}
int_4 getFmt (int_4 bits) {
  int_4 fmt = 0;
  if (bits<0) { fmt |= 0x8; bits=-bits; }
  if (bits==8)  fmt |= 0x1;
  if (bits==4)  fmt |= 0x2;
  if (bits==1)  fmt |= 0x3;
  if (bits==32) fmt |= 0x4;
  return fmt;
}
int_4 bits2fmt (int_4 bits) {
  int_4 fmt = 0;
  if (bits<0) { fmt |= 0x8; bits=-bits; }
  if (bits==8)  fmt |= 0x1;
  if (bits==4)  fmt |= 0x2;
  if (bits==1)  fmt |= 0x3;
  if (bits==32) fmt |= 0x4;
  return fmt;
}
int_4 form2fmt (char *form) {
  int_4 fmt = 0;
  if (form[1]=='I') fmt |= 0x0;
  if (form[1]=='B') fmt |= 0x1;
  if (form[1]=='N') fmt |= 0x2;
  if (form[1]=='P') fmt |= 0x3;
  if (form[1]=='L') fmt |= 0x4;
  if (form[0]=='C') fmt |= 0x8;
  return fmt;
}

int_4 scltbl[15] = { 0,0,0,0,0,0, 0x2A, 0x6A, 0xAA, 0x1AA, 0x2AA, 0x6AA, 0xAAA, 0x1AAA, 0x2AAA };

int_4 fftscale (int_4 shift, int_4 pow2) {
  int_4 i,j,k,s=shift;
#if CORE_IS_ICE
  int_4 scale = scltbl[pow2];
  for (i=16; i>0 && s>0; i-=2) {
    j = (scale>>i)&0x3;
    k = (j<s)?j:s;
    if (k>0) { j-=k; s-=k; scale = (scale&~(0x3<<i)) | (j<<i); }
  }
  /* printf("shift=%d pow=%d scltbl=%04x scale=%04x\n",shift,pow2,scltbl[pow2],scale); */
  return scale;
#else
  return shift;
#endif
}

/*
  routines for calling FPGA functions as an attached processor 
*/
int_4 core_config (void *handle, int_4 key, int_4 value) {
  return Core_config (handle,key,value);
}
int_4 core_init (void *handle) {
  HALO *halo = (HALO*)handle;
  if (halo->init!=NULL) return (*halo->init) (handle);
  printf("Trying to init an uninitialized HALO structure\n");
  return -1;
}
int_4 core_open (void *handle) {
  HALO *halo = (HALO*)handle;
  if (halo->open!=NULL) return (*halo->open) (handle);
  printf("Trying to open an uninitialized HALO structure\n");
  return -1;
}
int_4 core_process (void *handle, void *si, void *sip, void *so, void *sop) {
  HALO *halo = (HALO*)handle; int status=0;
  if (halo->work!=NULL) status = (*halo->work) (handle, si, sip, so, sop, 64); /* hardware dataflow method */
  else 			status = (*halo->process) (handle, si, so); 
  return status;
}
int_4 core_poll (void *handle) {
  HALO *halo = (HALO*)handle;
  return (*halo->poll) (handle);
}
int_4 core_close (void *handle) {
  HALO *halo = (HALO*)handle;
  return (*halo->free) (handle);
}
int_4 core_setfmt (void *handle, char *iform, char *oform) {
  int_4 fmts = (getFormat(iform)<<0) | (getFormat(oform)<<8);
  return Core_config (handle,1,fmts);
}
int_4 core_setscid (void *handle, int_4 scid) {
  return Core_config (handle,5,scid);
}
int_4 core_setmcid (void *handle, int_4 mcid) {
  return Core_config (handle,6,mcid);
}
int_4 core_set (void *handle, char *name, Value *value) {
  HALO *halo = (HALO*)handle;
  return (*halo->set) (handle,name,value);
}
int_4 core_get (void *handle, char *name, Value *value) {
  HALO *halo = (HALO*)handle;
  return (*halo->get) (handle,name,value);
}
int_4 core_setkey (void *handle, const char *name, void *value, int_4 size) {
  HALO *halo = (HALO*)handle;
  Value data;
  data.b = (int_1*)value;
  data.len = size;
  data.type = name[0];
  return (*halo->set) (handle,(char*)name,&data);
}
int_4 core_getkey (void *handle, const char *name, void *value, int_4 size) {
  HALO *halo = (HALO*)handle;
  Value data;
  data.b = (int_1*)value;
  data.len = size;
  data.type = name[0];
  return (*halo->get) (handle,(char*)name,&data);
}
real_8 core_s2d (char *s) {
  real_8 d; int_4 ls=strlen(s);
  int na = sscanf(s,"%lf",&d);
  if (na!=1) printf("Problem converting [%s] to a number\n",s);
  if (s[ls-1]=='K') d*=(1024);
  if (s[ls-1]=='M') d*=(1024*1024);
  if (s[ls-1]=='G') d*=(1024*1024*1024);
  return d;
}
int_4 core_setkeyfile (void *handle, const char *path) {
  int stat=-1,lval; float fval; double dval;
  char line[80],prefix[40],key[40],pkey[88],value[80],*s;
  FILE *fd;
  fd = fopen(path,"r");
  if(fd==NULL) { printf("Err opening parameter file=%s\n",path); return stat; }
  printf("File = %s\n",path);
  prefix[0]=0;
  for (stat=0; ; stat++) {
    if (fgets(line,80,fd)==NULL) break;
    s = strstr(line,"\n"); if (s!=NULL) s[0]=0; 
    s = strstr(line,"!"); if (s!=NULL) s[0]=0; 
    s = strstr(line,"="); if (s==NULL) continue; else s[0]=0;
    sscanf(line,"%s",key);
    sscanf(s+1,"%s",value);
         if (strncmp(value,"{",1)==0) sprintf(prefix,"%s_",key); 
    else if (strcmp(key,"NAME")==0||strcmp(key,"S:NAME")==0);
    else if (strncmp(key,"L:",2)==0) { lval=core_s2d(value); sprintf(pkey,"L:%s%s",prefix,key+2); core_setkey(handle,pkey,&lval,4); }
    else if (strncmp(key,"F:",2)==0) { fval=core_s2d(value); sprintf(pkey,"F:%s%s",prefix,key+2); core_setkey(handle,pkey,&fval,4); }
    else if (strncmp(key,"D:",2)==0) { dval=core_s2d(value); sprintf(pkey,"D:%s%s",prefix,key+2); core_setkey(handle,pkey,&dval,8); }
    else if (strncmp(key,"S:",2)==0) { sprintf(pkey,"A:%s%s",prefix,key+2); core_setkey(handle,pkey,(void*)value,strlen(value)); }
    else printf("Unhandled key=%s value=%s pair\n",key,value);
  }
  fclose(fd);
  return stat;
}

int_4 core_setkeytable (void *handle, char *stable) {
  int lval,i,j=0,ls; float fval; double dval;
  char line[80],prefix[40],pkey[40],*s,*key,*value;
  prefix[0]=0;
  i=0; if (stable[0]=='{'||stable[0]=='(') i++;
  ls=strlen(stable); if (stable[ls-1]=='}'||stable[ls-1]==')') ls--;
  for (;i<ls;i=j+1) {
    s = index(stable+i,','); j = (s==NULL)? ls : (s-stable); 
    strncpy(line,stable+i,j-i); line[j-i]=0;
    s = strstr(line,"="); if (s==NULL) continue; else s[0]=0;
    key = line; while (key[0]==' ') key++;
    value = s+1; while (value[0]==' ') value++;
         if (strncmp(value,"{",1)==0) sprintf(prefix,"%s_",key); 
    else if (strcmp(key,"NAME")==0||strcmp(key,"S:NAME")==0);
    else if (strlen(value)==0);	/* for status read only key */
    else if (strncmp(key,"L:",2)==0) { lval=core_s2d(value); sprintf(pkey,"L:%s%s",prefix,key+2); core_setkey(handle,pkey,&lval,4); }
    else if (strncmp(key,"F:",2)==0) { fval=core_s2d(value); sprintf(pkey,"F:%s%s",prefix,key+2); core_setkey(handle,pkey,&fval,4); }
    else if (strncmp(key,"D:",2)==0) { dval=core_s2d(value); sprintf(pkey,"D:%s%s",prefix,key+2); core_setkey(handle,pkey,&dval,8); }
    else if (strncmp(key,"S:",2)==0) { sprintf(pkey,"S:%s%s",prefix,key+2); core_setkey(handle,pkey,(void*)value,strlen(value)); }
    else printf("Unhandled key=%s value=%s pair\n",key,value);
  }
  return j;
}

#include <dlfcn.h>

/* C/C++ Interface */
void *core_alloc (char *cname, char *cimpl, void *context) {
  void *libhandle, *(*func)(char*); HALO *corehandle; char ctype=cimpl[0];
  char path[80],path2[80],sig[8],name[40],function[40],opt[16],optroot[40],*s,*config;
  char dconfig[8]="libCore"; 
  PICSTRUCT *p;
  int i;
  if (strcmp(cimpl,"ICEP")==0) {
    p = (PICSTRUCT *)context;
    config = dconfig;
  } else {
    p = NULL;
    config = (char*)context;
  }
  /* parse the option name */
  s = (char*)strchr(cname,';');
  if (s!=NULL) {
    i = (int)((long)s-(long)cname);
    strncpy(name,cname,i); name[i]=0;
    strcpy(opt,cname+i+1);
    strcat(opt,"ROOT");
  } else {
    strcpy(name,cname);
    strcpy(opt,"ICEROOT");
  }
  /* parse the core type */
       if (ctype=='C') strcpy(sig,"CPU");
  else if (ctype=='V') strcpy(sig,"VHS");
  else if (ctype=='I') strcpy(sig,"ICE");
  else if (ctype=='G') strcpy(sig,"GPU");
  else return 0;
  /* open the core library */
  strcpy(optroot,"/user/opt/icexxx");
  s=getenv(opt); if (s==NULL) printf("Error getting env var=%s\n",opt);
  if (s!=NULL) strcpy(optroot,s);
  sprintf(path,"%s/core/%s$%s_lin64.so",optroot,name,sig);
  sprintf(path2,"%s/core/Core%s_%s.so",optroot,sig,name);
  libhandle = dlopen(path, RTLD_LAZY|RTLD_GLOBAL);
  if (!libhandle) libhandle = dlopen(path2, RTLD_LAZY|RTLD_GLOBAL);
  if (!libhandle) { printf("Err opening library name=%s path=%s or path=%s\n",name,path,path2); return 0; }
  sprintf(function,"Core%s_%s_alloc",sig,name);	/* try multi-implementation format */
  func = (void *(*) (char*)) dlsym(libhandle, function);
  if (!func) { printf("Err opening function %s\n",function); return 0; }
  corehandle = (HALO*)(*func)(config);
  if (!corehandle) { printf("Err running function %s\n",function); return 0; }
  if (p!=NULL) corehandle->ph = p;
  core_init(corehandle);
  return corehandle;
}

char *f2c (char *fstr, int flen) {
  int i; 
  for (i=0; fstr[i]>' ' && i<flen; i++);
  fstr[i]=0;
  return fstr;
}

#ifdef _XMIDAS

#include "headers.h"
#include "hwlib.h"

void *core_alloc_xm (char *cname, const char *cimp, const char *cdev) {
  char sig[8],dev[40],flgs[40],config[128],typ,*s;
  int i,j,lc,cindex,pmi,ci,side,coreio;
  HEADER hwf;

  strcpy(sig,cimp);
  strcpy(dev,cdev);
  strcpy(config,"");

  coreio = (sig[3]=='X')?0:1;
  flgs[0]=0;

  s = strstr(dev,"|"); if (s!=NULL) { strcpy(flgs,s+1); s[0]=0; }
  s = strstr(dev,":"); if (s==NULL) strcat(dev,":11");

  /* parse for special implementation configuration */
  if (sig[0]=='I') {
    i = (int)((long)strchr(dev,':')-(long)dev);
    j=i+1; if (strncmp(dev+j,"CORE",4)==0) j+=4;
    cindex = atoi(dev+j);
    if (cindex<0) cindex=40-cindex;
    pmi = cindex/10; ci = cindex%10; side = (ci+1)%2+1;
    dev[i]=0; /* remove :port */
    i  = m_hwf_open(&hwf);
    lc = m_hwf_alias(&hwf,dev,config,128);
    i  = m_hwf_close(&hwf);
    sprintf(config+lc,",IPORT=PM0CORE%d,OPORT=PM%dCORE%d,COREIO=%d,%s",side,pmi,ci,coreio,flgs);
#if DEBUG
    printf("Alias=%s Port=%d Config=%s\n",dev,cindex,config);
#endif
  }
  return core_alloc (cname, sig, config);
}

/* Fortran Interface */
int_8 core_alloc_   (char *name, int ln) { return (int_8)core_alloc_xm(f2c(name,ln),"C","PIC1AUTO:11"); }
int_4 core_open_    (CORE *handle) { return core_open(handle); }
int_4 core_close_   (CORE *handle) { return core_close(handle); }
int_4 core_process_ (CORE *handle, void *si, void *sip, void *so, void *sop)    { return core_process(handle,si,sip,so,sop); }
int_4 core_setkey_  (CORE *handle, char *name,void *value,int_4 *size, int len) { return core_setkey(handle,f2c(name,len),value,*size); }
int_4 core_getkey_  (CORE *handle, char *name,void *value,int_4 *size, int len) { return core_getkey(handle,f2c(name,len),value,*size); }
int_4 core_setfmt_  (CORE *handle, char *iform, char *oform, int lif, int lof)  { return core_setfmt(handle,f2c(iform,lif),f2c(oform,lof)); }

#endif

