/////////////////////////////////////////////////////////////////
//  Picdrv.c - ICE-PIC Device handling routines for Windows OSes
//         Author:  Jeff Schoen
/////////////////////////////////////////////////////////////////

#define ICE_DRIVER_VERSION 338

#define _DOS 1
#define _WIN 1
#define _EEEI 1
 
// main WinDK header file
#include "WinDKC.h"
#include "WkRes.h"

// prototypes and structure definitions
#include "picdrv.h"

// message file include
#include "icepicmc.h"

typedef ULONGLONG int_8;
typedef int int_4;

// user interface and ioctl codes
#include "iceioctl.h"
#include "iceflex.h"
#include "iceconv.h"

#define MAXPIC 12
#define MAXIOBUF 512
#define MAXWCNT 0x40000000
#define CURRENTPID 1
 
#define UCB DEVICE_EXTENSION
typedef struct {
  UCB    *ucb;
  int	   pid;
  void   *usradr;
  int    psize;
  int    paddr;
} DMACHANNEL;
static int ndma=0;
static DMACHANNEL dma[MAXIOBUF]; 
static int_4 rstart, rsized, rsizem;

static ULONG RamStart=124,RamSizeDisk=0,RamSizeMap=4,MinLatency=0x80;
static WKBMD_INFORMATION RamBuffer;

static PDRIVER_OBJECT   pDriverObjectStatic;
#define dkInfo(A) KdMessage(A)
#define dkInfo1(A,B) KdMessage1(A,B)
#define dkInfo3(A,B,C,D) KdMessage3(A,B,C,D)

void KdMessage (char *string) {
  UNICODE_STRING tempStr;                
  WkStrConvertAnsi(&tempStr,string);
  WkLogErrorWithStrings(pDriverObjectStatic,CHECK_POINT,1,tempStr.Buffer); 
  WkStrFree(&tempStr);
}

void KdMessage1 (char *string, ULONG value) {
  UNICODE_STRING tempStr;                
  WkStrConvertAnsi(&tempStr,string);
  WkStrAppendInteger(&tempStr,value);
  WkLogErrorWithStrings(pDriverObjectStatic,CHECK_POINT,1,tempStr.Buffer); 
  WkStrFree(&tempStr);
}

void KdMessage3 (char *string, ULONG value1, ULONG value2, ULONG value3) {
  UNICODE_STRING tempStr,space;                
  WkStrConvertAnsi(&tempStr,string);
  WkStrConvertAnsi(&space," ");
  WkStrAppendInteger(&tempStr,value1);
  WkStrAppend(&tempStr,&space);
  WkStrAppendInteger(&tempStr,value2);
  WkStrAppend(&tempStr,&space);
  WkStrAppendInteger(&tempStr,value3);
  WkLogErrorWithStrings(pDriverObjectStatic,CHECK_POINT,1,tempStr.Buffer); 
  WkStrFree(&tempStr);
  WkStrFree(&space);
}


ULONG SetupCardbusController(PCHAR,ULONG);
NTSTATUS ReadWriteConfigSpace(PDEVICE_OBJECT,ULONG,PVOID,ULONG,ULONG);
NTSTATUS GetResources(PDEVICE_OBJECT,PUNICODE_STRING,PVOID);

void udelay (int usec) {
  LARGE_INTEGER delay={0,0};
  delay.QuadPart = -10*usec;
  KeDelayExecutionThread (KernelMode,TRUE,&delay);
}

ULONG icepic_mapio (UCB *ucb, int function, int *channel, int psize);
ULONG icepic_cancel (UCB *ucb, int reason);
ULONG icepic_setram (UCB *ucb, int start, int sized, int sizem);


#ifdef WINDK_ENV_WDM
///////////////////////////////////////////////////////////////////
//  DriverEntry - initial driver entry point for Icepic driver
//  Input:  pDriverObject - NT driver object
//          pRegistryPath - path to this drivers registry entries
//  Output: return() - NTSTATUS value
//  Notes:
///////////////////////////////////////////////////////////////////
NTSTATUS DriverEntry(IN  PDRIVER_OBJECT   pDriverObject,
                     IN  PUNICODE_STRING  pRegistryPath)
{
    NTSTATUS status;                 // return status
    WKREG_REGISTRY_PATH regPath;     // path into the registry

    pDriverObjectStatic = pDriverObject;
    dkInfo1("ICEPIC:  DriverEntry Version ",ICE_DRIVER_VERSION);

    // initialize pnp portion of driver.  Sets default handlers for
    status = WkDrvInitializePnpDriver(pDriverObject, pRegistryPath);
    if (!NT_SUCCESS(status)) return status;

    // setup the callback functions
    pDriverObject->MajorFunction[IRP_MJ_CLEANUP] = IcepicCleanup;
    pDriverObject->MajorFunction[IRP_MJ_WRITE] = IcepicDriverDispatch;
    pDriverObject->MajorFunction[IRP_MJ_READ] = IcepicDriverDispatch;
    pDriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = IcepicDriverDispatch;
    // a custom handler for unload is needed
    pDriverObject->DriverUnload = IcepicUnload;
    // plug and play add device
    WkDrvSetAddDeviceCallback(pDriverObject, IcepicAddDevice);

    // get the resources from the registry
    WkRegInitialize(pRegistryPath, &regPath);
    WkRegSetRelativePath(&regPath, L"Parameters");
    WkRegGetDword(&regPath, L"RamStart", &RamStart, &RamStart);
    WkRegGetDword(&regPath, L"RamSizeDisk", &RamSizeDisk, &RamSizeDisk);
    WkRegGetDword(&regPath, L"RamSizeMap", &RamSizeMap, &RamSizeMap);
    WkRegGetDword(&regPath, L"MinLatency", &MinLatency, &MinLatency);
    WkRegFree(&regPath);
    dkInfo3("Registry RAM Buffer",RamStart,RamSizeDisk,RamSizeMap);

    return STATUS_SUCCESS;
}

///////////////////////////////////////////////////////////////////////////
//  IcepicAddDevice - creates a new device object
//  Input:  pDriverObject   - our driver object
//          pPhysicalDeviceObject  - physical device object
//  Output: NTSTATUS code
//  Notes: Creates Device object but doesn't touch resources
///////////////////////////////////////////////////////////////////////////
NTSTATUS CALLBACK IcepicAddDevice(PDRIVER_OBJECT pDriverObject,
                                  PDEVICE_OBJECT pPhysicalDeviceObject)
{
    NTSTATUS        status;             // the return status
    PDEVICE_OBJECT  pDeviceObject;      // current device object
    ULONG           instance;           // instance number of the device 

    // since PnP can add and delete us in any order, search for an unused
    // device name (instead of just incrementing an instance number)
    // find the first instance number that is not already used
    // (alternately, use a device interface GUID)
    for (instance = 0; instance < ICEPIC_DEVICES_SUPPORTED; instance++) {
      // Create the device object
      status = IcepicCreateDevice(pPhysicalDeviceObject,pDriverObject, L"Icepic", 
               WkDrvGetDriverRegistryPath(pDriverObject),instance,&pDeviceObject);
      if (NT_SUCCESS(status)) {  	// try to create the next instance
        pDeviceObject->Flags &= ~DO_DEVICE_INITIALIZING;
        dkInfo("ICEPIC: AddDevice, returning STATUS_SUCCESS");
        return STATUS_SUCCESS;
      } else {			// delete the device
        if (NULL != pDeviceObject) IcepicDeleteDevice(pDeviceObject);
        // we did not successfully create a device, set the device pointer 
        pDeviceObject = NULL;
      }
    }

    // failed to create the device
    dkInfo("ICEPIC:  Failed to CreateDevice Icepic");
    WINDK_ASSERT(NT_SUCCESS(status));

    // the PnP AddDevice callback must always return STATUS_SUCCESS
    return STATUS_SUCCESS;
}

/////////////////////////////////////////////////////////////////////
//  IcepicCreateDevice - creates an initialized device object
//  Input:  pPhysicalDeviceObject - physical device object
//          pDriverObject         - driver object
//          pDeviceName           - name of this device
//          pRegistryPath         - path to the registry entries for this driver
//          DeviceNumber          - device number
//  Output: ppDeviceObject        - device object for this instance
//                                  of the device
//  Notes:
//////////////////////////////////////////////////////////////////////
NTSTATUS IcepicCreateDevice(IN  PDEVICE_OBJECT  pPhysicalDeviceObject,
                            IN  PDRIVER_OBJECT  pDriverObject,
                            IN  PWSTR           pDeviceName,
                            IN  PUNICODE_STRING pRegistryPath,
                            IN  ULONG           DeviceNumber,
                            OUT PDEVICE_OBJECT  *ppDeviceObject)
{
    NTSTATUS            status;                 // status return
    UNICODE_STRING      deviceName;             // the name of the device
    UNICODE_STRING      linkName;               // holds the link name
    UNICODE_STRING      tempStr;                // holds a temporary string
    PDEVICE_EXTENSION   ucb,pExtension;         // the device extension
    PWKDEV_INFORMATION  pDeviceInfo;            // device information

    // initialize the device name
    WkStrConvertUnicode(&deviceName, L"\\Device\\");
    WkStrConvertUnicode(&tempStr, pDeviceName);
    WkStrAppend(&deviceName, &tempStr);
    WkStrAppendInteger(&deviceName, DeviceNumber);
    WkStrFree(&tempStr);

    // create the device object
    status = IoCreateDevice (pDriverObject,sizeof(DEVICE_EXTENSION),
                &deviceName,ICEPIC_DEVICE_TYPE,0,FALSE,ppDeviceObject);
    if (!NT_SUCCESS(status)) {
        *ppDeviceObject = NULL;
        WkStrFree(&deviceName);
        dkInfo("ICEPIC: Unable to create device object");
        return status;
    }

    ucb = pExtension = (*ppDeviceObject)->DeviceExtension;  // get the new dev ext
    RtlZeroMemory(pExtension, sizeof(DEVICE_EXTENSION));  // zero the extension
    pDeviceInfo = &(pExtension->Information);      	  // get the WinDK info

    // initialize the device information
    status = WkDevInitialize(pPhysicalDeviceObject, pDeviceInfo, 
				*ppDeviceObject, &deviceName);
    // free the device name string
    WkStrFree(&deviceName);
    if (!NT_SUCCESS(status)) {
        dkInfo("ICEPIC: Unable to initialize device information");
        return status;
    }

    if (RamStart>=0) {		// set up the ram common buffer
      icepic_setram (pExtension, RamStart, RamSizeDisk, RamSizeMap);
      RamStart=-1;
    }

    // create a Win32 accessible name for the driver in the form of Icepic0
    WkStrConvertUnicode(&linkName, L"Icepic");
    WkStrAppendInteger(&linkName, DeviceNumber);
    status = WkDevCreateWin32Name(pDeviceInfo, &linkName);
    WkStrFree(&linkName);
    if (!NT_SUCCESS(status)) { dkInfo("ICEPIC: CreateWin32Name failed");return status; }

    return status;
}


/////////////////////////////////////////////////////////////////////
//  IcepicDeleteDevice - deletes the device
//  Input:  pDeviceObject - the device object to delete
//  Output: none
//  Notes:  none
/////////////////////////////////////////////////////////////////////
VOID IcepicDeleteDevice(IN  PDEVICE_OBJECT  pDeviceObject)
{
    PDEVICE_EXTENSION   pExtension;     // the device extension
    PWKDEV_INFORMATION  pDeviceInfo;    // device information

    // make sure the device object is non-null
    if (pDeviceObject == NULL) return;
    // get the extension
    pExtension = pDeviceObject->DeviceExtension;
    // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);
    // delete the device information
    WkDevFree(pDeviceInfo);
    // delete the device object
    IoDeleteDevice(pDeviceObject);
}


////////////////////////////////////////////////////////////////////
//  WkPnpSetupDeviceResources - called in response to
//          IRP_MJ_PNP:IRP_MN_START_DEVICE -- setup resources
//  Input:  pDeviceObject - device object for device whose
//          resources are being set up
//          pRawResourceList - raw resources
//          pTranslatedResourcesList - translated resources
//  Output: none
//  Notes:  this function must be provided by a pnp device
////////////////////////////////////////////////////////////////////
NTSTATUS WkPnpSetupDeviceResources(
                    PDEVICE_OBJECT pDeviceObject,
                    PCM_RESOURCE_LIST pRawResourceList,
                    PCM_RESOURCE_LIST pTranslatedResourceList)
{
    PDEVICE_EXTENSION   pExtension;                // the device extension
    NTSTATUS            status;                    // status code
    WKRES_INFORMATION   resources;                 // resource information
    PWKDEV_INFORMATION  pDeviceInfo;               // device information
    UNICODE_STRING      tempStr1;                  // used for logging errors
    UNICODE_STRING      tempStr2;                  // used for logging errors
    ULONG			data1, len1;		
    PCI_COMMON_CONFIG	pciconfig;

    // get the device extension from the device object
    pExtension = pDeviceObject->DeviceExtension;
    // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);

    // set up the resource list
    status = WkResInitializePnp(&resources,pTranslatedResourceList,pDeviceInfo);
    if (!NT_SUCCESS(status)) {
      dkInfo("ICEPIC: Unable to initialize resource information");
      return status;
    }

    // get the port and memory resources we need
    if (!WkResGetMemoryResource(&resources, 0, &pExtension->Registers)) {
      // missing memory address, fail the load
      dkInfo("ICEPIC: GetMemoryResource (0) failed");
      WkResFree(&resources);
      return STATUS_INSUFFICIENT_RESOURCES;
    }
    pExtension->Registers.Length = 4096;
    // done with resource gathering
    WkResFree(&resources);

    // map the port and memory addresses
    status = WkDevMapIoAddressPnp(&pExtension->Registers);
    if (!NT_SUCCESS(status)) {  // log the event
      WkStrConvertInteger(&tempStr1, pExtension->Registers.UnmappedAddress.LowPart, NULL);
      WkStrConvertInteger(&tempStr2, pExtension->Registers.Length, NULL);
      WkLogErrorWithStrings(pDeviceObject, FAILED_TO_MAP_ADDRESS, 3, 
              pDeviceInfo->DeviceName.Buffer,tempStr1.Buffer, tempStr2.Buffer);
      WkStrFree(&tempStr1);
      WkStrFree(&tempStr2);
      // unmap all previously mapped resources
      IcepicUnMapDeviceResources(pExtension);
      // return the status
      dkInfo("ICEPIC: MapIoAddress (Registers) Failed");
      return status;
    }

    // initialize the hardware
    status = IcepicInitializeBoard(pExtension);
    if (!NT_SUCCESS(status)) {
      // log the event
      WkLogErrorWithStrings(pDeviceObject, FAILED_TO_INITIALIZE_HARDWARE, 0);
      // unmap all previously mapped resources
      IcepicUnMapDeviceResources(pExtension);
      // return the status
      dkInfo("ICEPIC: IcepicInitializeBoard Failed");
      return status;
    }
    return status;
}

////////////////////////////////////////////////////////////////////
//  WkPnpFreeDeviceResources -- free any resources
//  Input:  pDeviceObject - the device object to delete
//          SurpriseRemove - TRUE if this remove was a surprise
//              remove
//  Output: none
//  Notes:  this function must be provided by a pnp device
////////////////////////////////////////////////////////////////////
NTSTATUS WkPnpFreeDeviceResources(IN PDEVICE_OBJECT pDeviceObject,
                                  IN BOOLEAN SurpriseRemove)
{
    PDEVICE_EXTENSION   pExtension;     // the device extension

    // get the extension
    pExtension = pDeviceObject->DeviceExtension;

    if (!SurpriseRemove) {
      // this is the last chance to program the hardware before the resources go away
      IcepicUnInitializeBoard(pExtension);
    }

    // unmap all previously mapped resources
    IcepicUnMapDeviceResources(pExtension);

    return STATUS_SUCCESS;
}

/////////////////////////////////////////////////////////////////////
//  IcepicUnMapDeviceResources - unmaps the device's resources
//  Input:  pExtension - the device extension
//  Output: none
//  Notes:  will only release resources that have already been allocated.
//          This function may be called at any time, and should not
//          access the card.
/////////////////////////////////////////////////////////////////////
VOID IcepicUnMapDeviceResources(IN  PDEVICE_EXTENSION pExtension)
{
    // unmap the memory addresses
    WkDevUnmapIoAddress(&pExtension->Registers);
}

/////////////////////////////////////////////////////////////////////
//  WkPnpDeleteDevice - deletes the device
//  Input:  pDeviceObject - the device object to delete
//  Output: none
//  Notes:  none
/////////////////////////////////////////////////////////////////////
VOID WkPnpDeleteDevice(IN  PDEVICE_OBJECT  pDeviceObject)
{
    IcepicDeleteDevice(pDeviceObject); // delete the device
}

//////////////////////////////////////////////////////////////////////
//  WkPowSavePowerRegisters - save power registers on power down
//  Input:  pDeviceObject - pointer to device object
//          pIrp - pointer to power down irp
//          deviceStateEntering - state device is entering for power down
//          systemStateEntering - state system is entering for power down
//  Output: NTSTATUS - if not success power down will fail
//  Notes:  device dependent call used on power down (IRP_MN_SET_POWER)
//          called by OnPowerSetPower handler -- called only when device
//          specific IRP received (as opposed to system) -- called only
//          if a transition occurs from PowerSystemWorking to a state
//          greater than PowerSystemWorking
//          derived class may optionally overload this
//          power information is stored in pIrpStack->Parameters.Power
//////////////////////////////////////////////////////////////////////
NTSTATUS WkPowSavePowerRegisters(PDEVICE_OBJECT pDeviceObject, 
                              PIRP           pIrp,
                              DEVICE_POWER_STATE deviceStateEntering, 
                              SYSTEM_POWER_STATE systemStateEntering)
{
    // TODO: saved data in hardware registers which are about to
    // be powered off
    return STATUS_SUCCESS;
}

//////////////////////////////////////////////////////////////////////
//  WkPowRestorePowerRegisters - restore hardware registers on power up
//  Input:  pDeviceObject - pointer to device object
//          pIrp - pointer to power down irp
//          deviceStateLeaving - state device was in before power up
//          systemStateLeaving - state system was in before power up
//  Output: NTSTATUS - if not success power down will fail
//  Notes:  device dependent call used on power up (IRP_MN_SET_POWER)
//          called by OnPowerSetPower handler -- called only when device
//          specific IRP received (as opposed to system) -- called only
//          when a transition occurs from a state greater than
//          PowerSystemWorking to a state <= PowerSystemWorking.
//          derived class may optionally overload this
//          power information is stored in pIrpStack->Parameters.Power
//////////////////////////////////////////////////////////////////////
NTSTATUS WkPowRestorePowerRegisters(PDEVICE_OBJECT pDeviceObject, 
                                    PIRP           pIrp,
                                    DEVICE_POWER_STATE deviceStateLeaving, 
                                    SYSTEM_POWER_STATE systemStateLeaving) 
{ 
    // TODO: write saved data to hardware registers
    return STATUS_SUCCESS; 
}

////////////////////////////////////////////////////////////////////
// WkPowGetDevicePowerStateFromSystemPowerState -  translates a
//         system power state into a device power state
// Input:  PowerState - system power state
// Output: device power state
// Notes:  default is <= PowerSystemWorking maps to On (PowerDeviceD0)
//         everything else maps to Off (PowerDeviceD3)
//////////////////////////////////////////////////////////
DEVICE_POWER_STATE WkPowGetDevicePowerStateFromSystemPowerState(
        SYSTEM_POWER_STATE PowerState)
{
    if (PowerState <= PowerSystemWorking) {
            // if the state is at least PowerSystemWorking then
            // we choose power state zero (full power) for
            // our device
            // Note: state values are smaller for more power
        return PowerDeviceD0;
    } else {
            // set power state to off
        return PowerDeviceD3;
    }    
}
////////////////////////////////////////////////////////////////////
//  WkPnpCleanupQueue - called in response to
//          IRP_MJ_PNP:IRP_MN_REMOVE_DEVICE -- clean up irp queue
//  Input:  pDeviceObject - the device object to delete
//  Output: none
//  Notes:  this function must be provided by a pnp device
////////////////////////////////////////////////////////////////////
VOID WkPnpCleanupQueue(IN PDEVICE_OBJECT pDeviceObject) 
{
    PDEVICE_EXTENSION  pExtension;        // device extension
    PWKDEV_INFORMATION pDeviceInfo;       // device information
    PIRP               pCurrentIrp;       // the current IRP
    PIO_STACK_LOCATION pCurrentIrpStack;  // our driver's information in this IRP
    
        // get device extension
    pExtension = pDeviceObject->DeviceExtension;
        // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);

        // get the current IRP
    pCurrentIrp = pDeviceObject->CurrentIrp;
        // NULL out current irp so the flush queue can't complete it
    pDeviceObject->CurrentIrp = NULL;
        // flush irp queue for this device
    WkDevFlushQueue(pDeviceInfo, NULL);
        // only cancel the current IRP if there indeed is a current IRP
    if (NULL != pCurrentIrp) {
            // TODO:: handle any outstanding I/O or DMA operations on the
            //  current IRP.  This case is usually caused by a surprise
            //  remove of your device.

            // cancel the current IRP
        WkDevCancelIrp(pDeviceInfo, pCurrentIrp, NULL);
    }
}

//////////////////////////////////////////////////////////////////////
//  IcepicUnload - main dispatch routine for unloading
//  Input:  pDriverObject - NT driver object
//  Output:
//  Notes:
//////////////////////////////////////////////////////////////////////
VOID IcepicUnload(IN  PDRIVER_OBJECT pDriverObject)
{
    dkInfo("IcepicUnload");
    // get rid of the DMA adapter
    if (RamBuffer.CommonBufferSize>0) WkBmdFree(&RamBuffer);
    // free pnp driver resources
    WkDrvFreePnpDriver(pDriverObject);
}


///////////////////////////////////////////////////////////////////
//  IcepicCleanup - Dispatch routine for Cleanup
//  Input:  pDeviceObject - NT device object
//          pIrp - IRP being dispatched
//  Output: return - NT status value
//  Notes:
///////////////////////////////////////////////////////////////////
NTSTATUS IcepicCleanup(IN  PDEVICE_OBJECT pDeviceObject,
                       IN  PIRP           pIrp)
{
    PDEVICE_EXTENSION     pExtension;     // the device extension
    PWKDEV_INFORMATION    pDeviceInfo;    // the WinDK device information
    PIO_STACK_LOCATION    pIrpStack;      // our portion of the IRP

        // get device extension
    pExtension = pDeviceObject->DeviceExtension;
        // get the WinDK device information
    pDeviceInfo = &(pExtension->Information);

        // increment the IRP count
    WkPnpIrpCountIncrement(pDeviceInfo);

        // initialize the IRP info field
    pIrp->IoStatus.Information = 0;
        // get our portion of the IRP
    pIrpStack = IoGetCurrentIrpStackLocation(pIrp);

        // flush all IRPs from the device queue
    WkDevFlushQueue(pDeviceInfo,
                    pIrpStack->FileObject);

        // finish completed IRPs
    WkPnpCompleteIrp(pDeviceInfo, pIrp, STATUS_SUCCESS, IO_NO_INCREMENT);
    return STATUS_SUCCESS;
}


#else	

///////////////////////////////////////////////////////////////////
//  DriverEntry - initial driver entry point for Icepic driver
//  Input:  pDriverObject - NT driver object
//          pRegistryPath - path to this drivers registry entries
//  Output: return() - NTSTATUS value
//  Notes:
///////////////////////////////////////////////////////////////////
NTSTATUS DriverEntry(IN  PDRIVER_OBJECT   pDriverObject,
                     IN  PUNICODE_STRING  pRegistryPath)
{
    NTSTATUS status = STATUS_SUCCESS;     // return status
    UNICODE_STRING deviceName;            // the name of the current device
    ULONG deviceNumber;                   // current device number
    ICEPIC_PCI_INFORMATION icepicContext; // context information
    PDEVICE_OBJECT pIcepic;               // the current Icepic device
    WKREG_REGISTRY_PATH regPath;          // path into the registry
        
    pDriverObjectStatic = pDriverObject;
    dkInfo1("ICEPIC:  DriverEntry Version ",ICE_DRIVER_VERSION);

    // setup the callback functions
    pDriverObject->MajorFunction[IRP_MJ_CREATE] = IcepicDriverDispatch;
    pDriverObject->MajorFunction[IRP_MJ_CLOSE] = IcepicDriverDispatch;
    pDriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = IcepicDriverDispatch;

    // a custom handler for unload is needed
    pDriverObject->DriverUnload = IcepicUnload;
    WkStrConvertUnicode(&deviceName, L"Icepic"); // create device name

    // override registry path
    WkStrConvertAnsi(pRegistryPath,
	"\\Registry\\Machine\\System\\CurrentControlSet\\Services");

    // get the resources from the registry
    WkRegInitialize(pRegistryPath, &regPath);
    WkRegSetRelativePath(&regPath, deviceName.Buffer);
    WkRegSetRelativePath(&regPath, L"Parameters");
    WkRegGetDword(&regPath, L"RamStart", &RamStart, &RamStart);
    WkRegGetDword(&regPath, L"RamSizeDisk", &RamSizeDisk, &RamSizeDisk);
    WkRegGetDword(&regPath, L"RamSizeMap", &RamSizeMap, &RamSizeMap);
    WkRegGetDword(&regPath, L"MinLatency", &MinLatency, &MinLatency);
    WkRegFree(&regPath);
    dkInfo3("Registry RAM Buffer",RamStart,RamSizeDisk,RamSizeMap);

    // set up the device context
    icepicContext.BusNumber = 0;
    icepicContext.SlotNumber = 0;

    // create a device for each card found in the system
    for (deviceNumber = 0; icepicContext.BusNumber < MAX_BUSES_TO_SEARCH;
         icepicContext.BusNumber++) {
        // keep going on this bus until we fail to find a card
        for (icepicContext.SlotNumber = 0;
             icepicContext.SlotNumber < PCI_MAX_DEVICES; deviceNumber++) {
            pIcepic = NULL; // null out our device pointer
            WkStrConvertUnicode(&deviceName, L"Icepic"); // create device name
            // create the device object
            status = IcepicCreateDevice(pDriverObject, deviceName.Buffer, 
                        pRegistryPath, deviceNumber, &pIcepic, &icepicContext);
            // free the string
            WkStrFree(&deviceName);
            if (!NT_SUCCESS(status)) {
                // failed to create the device, cleanup
                IcepicDeleteDevice(pIcepic);
                // go onto the next bus
                break;
            }
        }
    }

    if (deviceNumber != 0) {
        // if we found at least one device, then return success
        status = STATUS_SUCCESS;
    } else {
        // failed to create a device, cleanup
        KdPrint(("ICEPIC: Failed to create any devices\n"));
        status = STATUS_INSUFFICIENT_RESOURCES;
    }
    return status;
}

//////////////////////////////////////////////////////////////////////
//  IcepicUnload - main dispatch routine for unloading
//  Input:  pDriverObject - NT driver object
//  Output: NT status code
//  Notes:
//////////////////////////////////////////////////////////////////////
VOID IcepicUnload(IN  PDRIVER_OBJECT pDriverObject)
{
    PDEVICE_OBJECT  pCurrentDevice;     // the current device we are deleting

    KdPrint(("Icepic:  IcepicUnload\n"));
    // get rid of the DMA adapter
    if (RamBuffer.CommonBufferSize>0) WkBmdFree(&RamBuffer);
    // delete the device objects
    for(pCurrentDevice = pDriverObject->DeviceObject;
        pCurrentDevice != NULL;
        pCurrentDevice = pDriverObject->DeviceObject) {
        // delete each device we created
        IcepicDeleteDevice(pCurrentDevice);
    }
}

/////////////////////////////////////////////////////////////////////
//  IcepicCreateDevice - creates an initialized device object
//  Input:  pDriverObject         - driver object
//          pDeviceName           - name of this device
//          pRegistryPath         - path to the registry entries for this driver
//          DeviceNumber          - device number
//          pContext              - pointer to the current device creation context
//  Output: ppDeviceObject        - device object for this instance
//                                  of the device
//  Notes:
//////////////////////////////////////////////////////////////////////
NTSTATUS IcepicCreateDevice(IN  PDRIVER_OBJECT  pDriverObject,
                            IN  PWSTR           pDeviceName,
                            IN  PUNICODE_STRING pRegistryPath,
                            IN  ULONG           DeviceNumber,
                            OUT PDEVICE_OBJECT  *ppDeviceObject,
                            IN  PVOID           pContext)
{
    NTSTATUS            status;                 // status return
    UNICODE_STRING      deviceName;             // the name of the device
    UNICODE_STRING      linkName;               // holds the link name
    UNICODE_STRING      tempStr;                // holds a temporary string
    PDEVICE_EXTENSION   pExtension;             // the device extension
    PWKDEV_INFORMATION  pDeviceInfo;            // device information
    WKREG_REGISTRY_PATH regPath;                // path into the registry
    UNICODE_STRING      msgstring;

        // initialize the device name
    WkStrConvertUnicode(&deviceName, L"\\Device\\");
    WkStrConvertUnicode(&tempStr, pDeviceName);
    WkStrAppend(&deviceName, &tempStr);
    WkStrAppendInteger(&deviceName, DeviceNumber);
    WkStrFree(&tempStr);
        // create the device object
    status = IoCreateDevice (pDriverObject, sizeof(DEVICE_EXTENSION),
                &deviceName, ICEPIC_DEVICE_TYPE, 0, FALSE, ppDeviceObject);

    if (!NT_SUCCESS(status)) {
        *ppDeviceObject = NULL;
        WkStrFree(&deviceName);
        dkInfo("ICEPIC: Unable to create device object");
        return status;
    }

    // get the new device extension
    pExtension = (*ppDeviceObject)->DeviceExtension;
    // zero the extension
    RtlZeroMemory(pExtension, sizeof(DEVICE_EXTENSION));
    // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);
    // initialize the device information
    pExtension->BusNumber = ((PICEPIC_PCI_INFORMATION)pContext)->BusNumber;
    pExtension->SlotNumber = ((PICEPIC_PCI_INFORMATION)pContext)->SlotNumber;
    status = WkDevInitialize(pDeviceInfo, *ppDeviceObject, &deviceName);
    // free the device name string
    WkStrFree(&deviceName);
    if (!NT_SUCCESS(status)) {
        dkInfo("ICEPIC: Unable to initialize device information");
        return status;
    }

    // create a Win32 accessible name for the driver in the form of Icepic0
    WkStrConvertUnicode(&linkName, L"Icepic");
    WkStrAppendInteger(&linkName, DeviceNumber);
    status = WkDevCreateWin32Name(pDeviceInfo, &linkName);
    WkStrFree(&linkName);
    if (!NT_SUCCESS(status)) {
        dkInfo("ICEPIC: CreateWin32Name failed");
        return status;
    }

    // find our card and get our resources
    status = GetResources(*ppDeviceObject, pRegistryPath, pContext);
    return status;
}

/////////////////////////////////////////////////////////////////////
//  GetResources - finds the card and gets resources
//  Input:  pDeviceObject - the device object we are using
//          pRegistryPath - the registry path to place resources on
//          pContext - the context structure (for slot number)
//  Output: none
//  Notes:
/////////////////////////////////////////////////////////////////////
NTSTATUS GetResources(IN  PDEVICE_OBJECT    pDeviceObject,
                      IN  PUNICODE_STRING   pRegistryPath,
                      IN  PVOID             pContext)
{
    WKRES_INFORMATION   resources;              // used for assigning resources
    PDEVICE_EXTENSION   pExtension;             // the device extension
    PWKDEV_INFORMATION  pDeviceInfo;            // device information
    LONG                pciSlot;                // PCI slot number
    ULONG               busNumber;              // PCI bus number
    NTSTATUS            status;                 // intermediate status
    UNICODE_STRING	tempStr1, tempStr2;
 
    // get the device extension
    pExtension = pDeviceObject->DeviceExtension;
    // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);
    // get the slot and bus number
    pciSlot = ((PICEPIC_PCI_INFORMATION) pContext)->SlotNumber;
    busNumber = ((PICEPIC_PCI_INFORMATION) pContext)->BusNumber;
    // we need to go get our resources
    // this is a PCI card, look for it on the bus
    status = WkResInitialize(&resources, PCIBus, busNumber, pDeviceInfo, 0);
    if (!NT_SUCCESS(status))
      {
        dkInfo("ICEPIC: INITIALIZE RESOURCES");
        WkResFree(&resources);
        return STATUS_DEVICE_NOT_CONNECTED;
      }
    if ((pciSlot = WkResFindPciCard(&resources, VENDOR_ID, DEVICE_ID, 0, pciSlot)) < 0) {
        // device not found, return failure
        WkResFree(&resources);
        return STATUS_DEVICE_NOT_CONNECTED;
    }
    // we found the device, bump up the PCI slot position
    // so the next device created will find the next adapter on the bus
    ((PICEPIC_PCI_INFORMATION) pContext)->SlotNumber = pciSlot + 1;

    // get the cards resources from its PCI configuration space
    WkStrConvertAnsi(&tempStr1,
	"\\Registry\\Machine\\System\\CurrentControlSet\\Services\\Icepic");

    status = WkResAssignPciCardsResources(&resources, &tempStr1, VENDOR_ID, DEVICE_ID, pciSlot, 0);
    
    if (!NT_SUCCESS(status)) {
        // could not obtain resources
        dkInfo("ICEPIC: WkResAssignPciCardsResources failed");
        WkResFree(&resources);
        return status;
    }

    // get the port and memory resources we need
    if (!WkResGetMemoryResource(&resources, 0, &pExtension->Registers)) {
        // missing memory address, fail the load
        dkInfo("ICEPIC: GetMemoryResource (0) failed");
        WkResFree(&resources);
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    // done with resource gathering
    WkResFree(&resources);

    // map the port and memory addresses
    status = WkDevMapIoAddress(PCIBus, busNumber, &pExtension->Registers);
    if (!NT_SUCCESS(status)) {
        // log the event
        WkStrConvertInteger(&tempStr1, pExtension->Registers.UnmappedAddress.LowPart, NULL);
        WkStrConvertInteger(&tempStr2, pExtension->Registers.Length, NULL);
        WkLogErrorWithStrings(pDeviceObject, FAILED_TO_MAP_ADDRESS, 3, 
         pExtension->Information.DeviceName.Buffer, tempStr1.Buffer, tempStr2.Buffer);
        WkStrFree(&tempStr1);
        WkStrFree(&tempStr2);
        // return the status
        dkInfo("ICEPIC: MapIoAddress (Memory0) Failed");
        return status;
    }

    // initialize the hardware
    status = IcepicInitializeBoard(pExtension);
    if (!NT_SUCCESS(status)) {
        // log the event
        WkLogErrorWithStrings(pDeviceObject, FAILED_TO_INITIALIZE_HARDWARE, 0);
        // return the status
        dkInfo("ICEPIC: IcepicInitializeBoard Failed");
        return status;
    }

    if (RamStart!=0) {		// set up the ram common buffer
      icepic_setram (pExtension, RamStart, RamSizeDisk, RamSizeMap);
      RamStart=0;
    }

    return STATUS_SUCCESS;
}

/////////////////////////////////////////////////////////////////////
//  IcepicDeleteDevice - deletes the device
//  Input:  pDeviceObject - the device object to delete
//  Output: none
//  Notes:  none
/////////////////////////////////////////////////////////////////////
VOID IcepicDeleteDevice(IN  PDEVICE_OBJECT  pDeviceObject)
{
    PDEVICE_EXTENSION   pExtension;     // the device extension
    PWKDEV_INFORMATION  pDeviceInfo;    // device information

    // make sure the device object is non-null
    if (pDeviceObject == NULL) return; 
    // get the extension
    pExtension = pDeviceObject->DeviceExtension;
    // get the WinDK information from the device extension
    pDeviceInfo = &(pExtension->Information);

    // unmap all previously mapped resources
    IcepicUnMapDeviceResources(pExtension);

    // delete the device information
    WkDevFree(pDeviceInfo);

    // delete the device object
    IoDeleteDevice(pDeviceObject);
}

/////////////////////////////////////////////////////////////////////
//  IcepicUnMapDeviceResources - unmaps the device's resources
//  Input:  pExtension - the device extension
//  Output: none
//  Notes:  will only release resources that have already been allocated.
//          This function may be called at any time, and should not
//          access the card.
/////////////////////////////////////////////////////////////////////
VOID IcepicUnMapDeviceResources(IN  PDEVICE_EXTENSION pExtension)
{
    // unmap the memory addresses
    WkDevUnmapIoAddress(&pExtension->Registers);
}

//////////////////////////////////////////////////////////////////////
//  IcepicInitializeBoard - initialize the hardware
//  Input:  pExtension - the device extension
//  Output: return - NTSTATUS code
//  Notes:  return STATUS_SUCCESS if the hardware initialized
//////////////////////////////////////////////////////////////////////
NTSTATUS IcepicInitializeBoard(IN  PDEVICE_EXTENSION pExtension)
{
    return STATUS_SUCCESS;
}

#endif


///////////////////////////////////////////////////////////////////
//  IcepicDriverDispatch - Icepic dispatch entry point
//  Input:  pDeviceObject - NT device object
//          pIrp - IRP being dispatched
//  Output: return - NT status value
//  Notes:  handles IRP_MJ_WRITE, IRP_MJ_READ and IRP_MJ_DEVICE_CONTROL
///////////////////////////////////////////////////////////////////
NTSTATUS IcepicDriverDispatch(IN  PDEVICE_OBJECT pDeviceObject,
                              IN  PIRP           pIrp)
{
    PDEVICE_EXTENSION     pExtension;     // the device extension
    PWKDEV_INFORMATION    pDeviceInfo;    // the WinDK device information
    PIO_STACK_LOCATION    pIrpStack;      // our portion of the IRP
    NTSTATUS              status;         // return status
	
    // get device extension
    pExtension = pDeviceObject->DeviceExtension;
    // get the WinDK device information
    pDeviceInfo = &(pExtension->Information);

#ifdef WINDK_ENV_WDM
    // make sure it's ok with PnP to take an IRP right now if so increment IRP count
    if (!WkPnpIrpPrologue(pDeviceInfo, pIrp, &status)) return status;
#endif

    // reset the status code
    status = STATUS_NOT_IMPLEMENTED;

    // initialize the IRP info field
    pIrp->IoStatus.Information = 0;
    // get our portion of the IRP
    pIrpStack = IoGetCurrentIrpStackLocation(pIrp);

    // dispatch based on major function code
    switch (pIrpStack->MajorFunction) {

      case IRP_MJ_DEVICE_CONTROL:
        if (pIrpStack->Parameters.DeviceIoControl.IoControlCode == ICEPIC_IOCTL)
        status = IcepicIoctl (pExtension,pIrp,pIrpStack);
        break;
      case IRP_MJ_CREATE:
      case IRP_MJ_CLOSE:
        status = STATUS_SUCCESS;
        break;
      default:
        status = STATUS_NOT_IMPLEMENTED;
        break;
    }

    if (status != STATUS_PENDING) {   // complete finished requests
#ifdef WINDK_ENV_WDM
      WkPnpCompleteIrp(pDeviceInfo, pIrp, status, IO_NO_INCREMENT);
#else
      WkDevCompleteIrp(pIrp, status, IO_NO_INCREMENT);
#endif
    }

    return status;
}



static int lastoffset;
#if _ALPHA
#define _RD(A,B) { B=READ_REGISTER_ULONG((PULONG)(((PUCHAR)ucb->Registers.pAddress)+(A))); } 
#define _WR(A,B) { WRITE_REGISTER_ULONG((PULONG)(((PUCHAR)ucb->Registers.pAddress)+(A)),B); } 
#else
#define _RD(A,B) { B=*(PULONG)(((PUCHAR)ucb->Registers.pAddress)+(A));} 
#define _WR(A,B) { *(PULONG)(((PUCHAR)ucb->Registers.pAddress)+(A))=(ULONG)B;} 
#endif

#define ESUCCESS STATUS_SUCCESS
#define EINVAL STATUS_INVALID_PARAMETER
#define ENOMEM STATUS_INVALID_PARAMETER

#define PIC_FUNCTION p->function
#define PIC_OFFSET   p->offset
#define PIC_BUFADDR  p->bufaddr
#define PIC_BYTES    p->bytes
#define PIC_STATUS   p->status

///////////////////////////////////////////////////////////////////
//  IcepicIoctl - performs a user request
//  Input:  pExtension - the device information
//          pIrp - pointer to the IRP
//          pIrpStack - IRP stack
//  Output: return - NT status code
//  Notes:  supports METHOD_BUFFERED ioctls only
///////////////////////////////////////////////////////////////////
NTSTATUS IcepicIoctl(IN  DEVICE_EXTENSION   *ucb,
                     IN  PIRP                pIrp,
                     IN  PIO_STACK_LOCATION  pIrpStack)
{
  int		i,j,k,ii,iout;		/* Local counters */
  int		status,stat;		/* Local function return status */
  int		chan;			/* Current dma channel */
  PICIOCTL	*p,ploc;		/* pic ioctl handle structure */
  ULONG		worda, wordb;		/* load IOC temps */
  ULONG		*datain, *dataout;	/* data buffer */
  int_4		msize,maddr;		/* word temps */
  ULONGLONG	usize,uaddr,uoffs;

  pIrp->IoStatus.Information = 0;
  if (pIrpStack->Parameters.DeviceIoControl.InputBufferLength<sizeof(PICIOCTL)) 
	return STATUS_INVALID_PARAMETER;

  p = (PICIOCTL *)pIrp->AssociatedIrp.SystemBuffer;

  i = PIC_BYTES;			/* normal|default completion status */
  status = ESUCCESS;	 		/* assume success */
  iout = sizeof(PICIOCTL);
  datain = dataout = (PULONG)((PUCHAR)pIrp->AssociatedIrp.SystemBuffer+iout);

  switch (PIC_FUNCTION) {

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=ii=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
        else break;
        for (k=0; k<j; k++) _RD(REG_FIFO, dataout[ii++]);  
        i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      _RD (PIC_OFFSET, dataout[0]);		
    }
    iout += i;
    break;

  case IOCTL_WRITE:			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=ii=0; i<PIC_BYTES; ) {
        _RD (REG_MCSR,stat);
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6;
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4;
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
        else break;
        for (k=0; k<j; k++) _WR(REG_FIFO, datain[ii++]);  
        i += (j*4);
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
      status = EINVAL;
    else {
      if (PIC_OFFSET==lastoffset+4) _WR(REG_PBAR+8,0);
      _WR(PIC_OFFSET,datain[0]);
      lastoffset=PIC_OFFSET;
    }
    break;

  case IOCTL_ACMD: 			/* command to SHARC processor */

    for (k=4; k>=0; k--) _WR(REG_MBX+k*4,datain[k]);
    for (i=0,j=0; j<32; j++) { 	/* cant wait forever in kernel mode */
      _RD (REG_MCSR,stat);
      if ( (stat&MCSR_IMB1F) != 0x0 ) {
        _RD (REG_IMB1, dataout[0]);		
        i=4; break;
      }
      if (j>4) udelay(100);
    }
    iout += i;
    break;

  case IOCTL_RALLOC: 			/* allocate contiguous RAM */

    status = icepic_setram (ucb, 
	(int)PIC_OFFSET, (int)PIC_BUFADDR, (int)PIC_BYTES);
    if (status==ESUCCESS) i=dma[1].paddr; else i=0;

    break;

  case IOCTL_MAP:			/* map user buffer to PCI */

    i = 0;
    /* find an open dma handler slot */
    if (ndma==0 || ndma+1>=MAXIOBUF) status=EINVAL; 

    msize = phys2word(PIC_BYTES);

    /* Validate user buffer */
    if ( (msize <= 0) || (msize > MAXWCNT) ) status = EINVAL;
    if (status!=ESUCCESS) break;

    /* map buffer into PCI bus space */
    status = icepic_mapio (ucb, IOCTL_MAP, &chan, msize);

    if (status==ESUCCESS) {
      usize = word2long(msize);
      uoffs = word2long(dma[chan].paddr);
      PIC_OFFSET = word2phys(dma[chan].paddr);
#ifdef WINDK_ENV_WDM
      status = WkDevMapMemoryToUserSpacePnp ((ULONGLONG)uoffs,
			(ULONG)usize,&((PVOID)PIC_BUFADDR));
#else
      status = WkDevMapMemoryToUserSpace ((ULONGLONG)uoffs,
		(ULONG)usize,PCIBus,ucb->BusNumber,&((PVOID)PIC_BUFADDR),(PVOID)0);
#endif
      if (status==ESUCCESS) i=word2phys(dma[chan].paddr); else i=0;
    }
    break;
    
  case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

    msize = phys2word(PIC_BYTES);
    maddr = phys2word(PIC_OFFSET);

    /* is it in the table of driver mapped PCI memory */
    for (chan=0; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr &&
          maddr+msize <= dma[chan].paddr+dma[chan].psize) break;
    }
    status = WkDevUnmapUserMemory ((PVOID)PIC_BUFADDR,Internal,0,NULL);
    if (chan>=ndma) { status = EINVAL; break; }

    /* unmap user buffer from PCI bus space */
    status = icepic_mapio (ucb, IOCTL_UNMAP, &chan, -1);
    break;

  case IOCTL_QMAP: 			/* query user buffer map */

    msize = phys2word(PIC_BYTES);
    maddr = phys2word(PIC_OFFSET);

    /* is it in the table of driver mapped PCI memory */
    for (chan=0; chan<ndma; chan++) {
      if (maddr >= dma[chan].paddr &&
          maddr+msize <= dma[chan].paddr+dma[chan].psize) break;
    }
    if (chan==ndma) i=0;
    else if (PIC_BYTES!=0) i=PIC_BYTES;
    else i=word2phys(dma[chan].paddr+dma[chan].psize-maddr);
    break;

  case IOCTL_LIOC:                      /* load IOC words */

    for (i=ii=0; i<PIC_BYTES; i+=4) {
      k = PIC_BYTES-i; if (k>4) k=4;
      worda = datain[ii++];
      for (j=0; j<(k*8); j++) {
        wordb = (worda>>j)&0x1;
        _WR(REG_IOC,wordb);
      }
    }
    break;

  case IOCTL_VERSION:                   /* get driver version */

    dataout[0] = ICE_DRIVER_VERSION;
    iout += i;
    break;

  default: 				/* handle unknown function codes */
    status = EINVAL;
    break;
  }

  PIC_STATUS = i;
  pIrp->IoStatus.Information = iout;
  return(status);
}

ULONG icepic_setram (UCB *ucb,
			int start, int sized, int sizem) {
  ULONG rambytes,status;
  if (start!=-1) rstart = start * 0x40000;      /* RAM start in words */
  if (sized!=-1) rsized = sized * 0x40000;      /* RAMDISK size in words */
  if (sizem!=-1) rsizem = sizem * 0x40000;      /* DYNRAM size in words */

  /* dkInfo3("Physical RAM Alloc ",rstart,rsized,rsizem); */
  if (rstart==0) {
    rambytes = ((ULONG)(rsized+rsizem))<<2;
    // setup the DMA description - mostly use defaults
    WkBmdInitialize(&ucb->Information,&RamBuffer);
    // set width and maximum length
    RamBuffer.DmaDescription.DmaWidth = Width32Bits;
    RamBuffer.DmaDescription.MaximumLength = rambytes;
    status = WkBmdSetupCommonBuffer(&RamBuffer,rambytes,FALSE);
    dkInfo3("DMA Common ",(ULONG)RamBuffer.CommonBufferPhysicalAddress.LowPart,
		(ULONG)RamBuffer.CommonBufferSize, rambytes);
    rstart = (int)RamBuffer.CommonBufferPhysicalAddress.LowPart;
    if (!NT_SUCCESS(status)) {
      // unmap all previously mapped resources
      WkBmdFree(&RamBuffer);
      // return the status
      dkInfo("ICEPIC: SetupDmaAdapter failed");
      rstart = rsized = rsizem = 0;
      return EINVAL;
    }
  }
  else {
    RamBuffer.CommonBufferSize = 0;
  }
  dma[0].paddr=rstart;
  dma[0].psize=rsized;
  dma[0].pid=0;
  dma[1].paddr=rstart+rsized+rsizem;
  dma[1].psize=0;
  dma[1].pid=0;
  ndma=1;
  return ESUCCESS;
}


/************************************/
/* pic cancel routine               */
/************************************/
ULONG icepic_cancel (UCB *ucb, int reason)
{
  ULONG status; int reset,chan;
  reset = MCSR_RESET;

  if (reason == 1) {	/* close */
    for (chan=ndma-1; chan>=1; chan--) {
      if (dma[chan].pid == CURRENTPID && dma[chan].paddr != 0) {
        dkInfo ("pic_cancel - stray DMA - resetting card");
        /* unexpected termination - reset card to avoid stray DMAs */
	_WR (REG_MCSR, reset);
	/* deallocate/unmap stray buffer */
	status = icepic_mapio (ucb, IOCTL_UNMAP, &chan, -1);
      }
    }
  }
  return(ESUCCESS); 
}



/************************************/
/* pic mapio routine                */
/************************************/
ULONG icepic_mapio (UCB *ucb, int function, int *channel, int psize)
{
  ULONG status;
  int i,n,chan;

  status = ESUCCESS;	/* assume success */

  switch (function) {

  case IOCTL_MAP:

    for (i=1; i<=ndma; i++) if (dma[i].paddr-
	(dma[i-1].paddr+dma[i-1].psize) >= psize) goto ALLOCATE;

    return ENOMEM; /* out of room */

    ALLOCATE:
    n=i; ndma++;
    for (i=ndma; i>n; i--) dma[i] = dma[i-1];
    dma[n].paddr = dma[n-1].paddr+dma[n-1].psize;
    dma[n].psize = psize;
    dma[n].pid = CURRENTPID;
//    wprintf("DMA map psize=%d n=%d paddr=%x %d\n",psize,n,dma[n].paddr,ndma);
    *channel = n;
    break;

  case IOCTL_UNMAP:

    chan = *channel;
    if (psize<0 && chan>=0) psize = dma[chan].psize;
//    wprintf("DMA unmap psize=%d %d\n",psize,ndma);
    for (i=chan; i<ndma; i++) dma[i] = dma[i+1];
    ndma--;
    break;

  default:
    status = EINVAL;
  }
#if DEBUG
  for (i=0; i<=ndma; i++)
//  wprintf("DMA table i=%d paddr=%x psize=%d \n",i,dma[i].paddr,dma[i].psize);
#endif
  return (status);
}

#ifdef WINDK_ENV_WDM

//////////////////////////////////////////////////////////////////////
//  IcepicInitializeBoard - initialize the hardware
//  Input:  pExtension - the device extension
//  Output: return - NTSTATUS code
//  Notes:  return STATUS_SUCCESS if the hardware initialized
//////////////////////////////////////////////////////////////////////
NTSTATUS IcepicInitializeBoard(IN DEVICE_EXTENSION *ucb)
{
    ULONG value1,value2;

    // disable auto power control
    ucb->Information.PnpInfo.PowerControlOnOpen = 0;

    // fix PCI mastering registers - in case BIOS falls flat 
    _RD(REG_PCSR,value1); value2 = value1|0x6;
    if (value2!=value1) { _WR(REG_PCSR,value2);
    dkInfo3("PCI Init CSR:  ",REG_PCSR,value1,value2); }

    _RD(REG_PLAT,value1); value2 = (value1&0xFFFF00FF)|(MinLatency<<8);
    if (((value1>>8)&0xFF)<MinLatency) { _WR(REG_PLAT,value2);
    dkInfo3("PCI Init LAT:  ",REG_PLAT,value1,value2); }

    SetupCardbusController("\\Device\\Pcmcia0",MinLatency);
    SetupCardbusController("\\Device\\Pcmcia1",MinLatency);

    return STATUS_SUCCESS;
}

//////////////////////////////////////////////////////////////////////
//  IcepicUnInitializeBoard - uninitialize the hardware
//  Input:  pExtension - the device extension
//  Output: none
//  Notes:
//////////////////////////////////////////////////////////////////////
VOID IcepicUnInitializeBoard(IN DEVICE_EXTENSION *ucb)
{
}

ULONG SetupCardbusController(PCHAR name, ULONG latency)
{
  NTSTATUS status;
  PFILE_OBJECT pFileObject;
  PDEVICE_OBJECT pDeviceObject;
  PCI_COMMON_CONFIG pcicom;
  UNICODE_STRING DeviceName; 

  WkStrConvertAnsi(&DeviceName,name);
  status = IoGetDeviceObjectPointer(&DeviceName, FILE_ALL_ACCESS,
                                      &pFileObject, &pDeviceObject);
  WkStrFree(&DeviceName);
  dkInfo3("PCI Cardbus Device:  ",(ULONG)status,0,(ULONG)pDeviceObject);
  if (status!=STATUS_SUCCESS) return status;

  status = ReadWriteConfigSpace (pDeviceObject,0,(PVOID)(&pcicom.LatencyTimer),13,1);
  dkInfo3("PCI Read Cardbus Bridge Lat:  ",(ULONG)pcicom.LatencyTimer,(ULONG)status,0);
  if (pcicom.LatencyTimer>latency) return STATUS_SUCCESS;

  pcicom.LatencyTimer = (UCHAR)latency;
  status = ReadWriteConfigSpace (pDeviceObject,1,(PVOID)(&pcicom.LatencyTimer),13,1);
  dkInfo3("PCI Write Cardbus Bridge Lat:  ",(ULONG)pcicom.LatencyTimer,(ULONG)status,0);

  return STATUS_SUCCESS;
}


NTSTATUS ReadWriteConfigSpace(
      IN PDEVICE_OBJECT DeviceObject,
      IN ULONG          ReadOrWrite, // 0 for read 1 for write
      IN PVOID          Buffer,
      IN ULONG          Offset,
      IN ULONG          Length
      )
  {
      KEVENT event;
      NTSTATUS status;
      PIRP irp;
      IO_STATUS_BLOCK ioStatusBlock;
      PIO_STACK_LOCATION irpStack;
      PDEVICE_OBJECT targetObject;

      PAGED_CODE();

      KeInitializeEvent( &event, NotificationEvent, FALSE );

      targetObject = IoGetAttachedDeviceReference( DeviceObject );

      irp = IoBuildSynchronousFsdRequest( IRP_MJ_PNP,
                                          targetObject,
                                          NULL,
                                          0,
                                          NULL,
                                          &event,
                                          &ioStatusBlock );

      if (irp == NULL) {
          status = STATUS_INSUFFICIENT_RESOURCES;
          goto End;
      }

      irpStack = IoGetNextIrpStackLocation( irp );

      if (ReadOrWrite == 0) {
          irpStack->MinorFunction = IRP_MN_READ_CONFIG;
      }else {
          irpStack->MinorFunction = IRP_MN_WRITE_CONFIG;
      }

      // PCI_WHICHSPACE_CONFIG = 0 on w9x where it is not defined yet
      irpStack->Parameters.ReadWriteConfig.WhichSpace = PCI_WHICHSPACE_CONFIG;
      irpStack->Parameters.ReadWriteConfig.Buffer = Buffer;
      irpStack->Parameters.ReadWriteConfig.Offset = Offset;
      irpStack->Parameters.ReadWriteConfig.Length = Length;

      // 
      // Initialize the status to error in case the bus driver does not 
      // set it correctly.
      // 

      irp->IoStatus.Status = STATUS_NOT_SUPPORTED ;

      status = IoCallDriver( targetObject, irp );

      if (status == STATUS_PENDING) {

          KeWaitForSingleObject( &event, Executive, KernelMode, FALSE, NULL );
          status = ioStatusBlock.Status;
      }

  End:
      // 
      // Done with reference
      // 
      ObDereferenceObject( targetObject );

      return status;

  } 

#endif

/*
typedef struct _PCI_COMMON_CONFIG {
    USHORT  VendorID;                   // (ro)
    USHORT  DeviceID;                   // (ro)
    USHORT  Command;                    // Device control
    USHORT  Status;
    UCHAR   RevisionID;                 // (ro)
    UCHAR   ProgIf;                     // (ro)
    UCHAR   SubClass;                   // (ro)
    UCHAR   BaseClass;                  // (ro)
    UCHAR   CacheLineSize;              // (ro+)
    UCHAR   LatencyTimer;               // (ro+)
    UCHAR   HeaderType;                 // (ro)
    UCHAR   BIST;                       // Built in self test
}
*/

