/****************************************************************/
/*  ICE-PIC Device Driver for DEC-OpenVMS                       */
/*                                                              */
/*  Author:    Jeff Schoen - Innovative Computer Engineering    */
/*  Original:  John Hill - Zeta Associates                      */
/*                                                              */
/****************************************************************/
 
#pragma module PICDRIVER "REV 3.38"
#define ICE_DRIVER_VERSION 338

/*****************/
/* Include Files */
/*****************/
#include <ccbdef.h>             /* Channel control block */
#include <crbdef.h>             /* Controller request block */
#include <cramdef.h>            /* Controller register access method */
#include <dcdef.h>              /* Device codes */
#include <ddbdef.h>             /* Device data block */
#include <ddtdef.h>             /* Driver dispatch table */
#include <devdef.h>             /* Device characteristics */
#include <dptdef.h>             /* Driver prologue table */
#include <dyndef.h>		/* Dynamic memory block types */
#include <fdtdef.h>             /* Function decision table */
#include <fkbdef.h>             /* Fork block */
#include <idbdef.h>             /* Interrupt data block */
#include <iocdef.h>             /* IOC constants */
#include <iodef.h>              /* I/O function codes */
#include <irpdef.h>             /* I/O request packet */
#include <ka0602def.h>          /* DEC 2000 Model 300 AXP specific defs  */
#include <lpdef.h>              /* Line printer definitions */
#include <orbdef.h>             /* Object rights block */
#include <pcbdef.h>             /* Process control block */
#include <msgdef.h>             /* System-wide mailbox message codes */
#include <mtxdef.h>             /* Longword mutex */
#include <ssdef.h>              /* System service status codes */
#include <stsdef.h>             /* Status value fields */
#include <ucbdef.h>             /* Unit control block */
#include <vecdef.h>             /* IDB interrupt transfer vector */
#include <vadef.h>		/* Virtual Address definitions */


/**********************************/
/* Function Prototype Definitions */
/**********************************/
#include <exe_routines.h>       /* Prototypes for EXE$ and EXE_STD$ routines */
#include <ioc_routines.h>       /* Prototypes for IOC$ and IOC_STD$ routines */
#include <sch_routines.h>       /* Prototypes for SCH$ and SCH_STD$ routines */
#include <erl_routines.h>	/* Prototypes for ERL$ and ERL_STD$ routines */
#include <com_routines.h>	/* Prototypes for COM$ and COM_STD$ routines */


/***********************************/
/* Device Driver Macro Definitions */
/***********************************/
#include <vms_drivers.h>        /* Device driver support macros */

/******************************/
/* DEC C Function Definitions */
/******************************/
#include <builtins.h>           /* OpenVMS AXP specific C builtin functions */
#include <string.h>             /* String routines provided by "kernel CRTL" */
#include <stddef.h>		/* Common C definitions */


/***********************/
/* General Definitions */
/***********************/
#define	true	(1==1)		/* True  in any reasonable implementation */
#define	false	(1!=1)		/* False in any reasonable implementation */

#define	$SUCCESS(a)	(((a) & 1) == 1)
#define	$FAILURE(a)	(((a) & 1) == 0)

#define	ABS(a) (((a)&0X80000000) ? -(a) : (a))

typedef	unsigned char	byte;
typedef unsigned short	word;
typedef	unsigned long	longword;
typedef uint64 		quadword;


/************************************/
/* VMS Page Size External Variables */
/************************************/
extern	longword	MMG$GL_BWP_MASK;	/* Page offset mask */
extern	longword	MMG$GL_PAGE_SIZE;	/* Page size in bytes */
extern	longword	MMG$GL_VA_TO_VPN;	/* Number bits to shift */
extern	longword	MMG$GL_SPTBASE;		/* System Page Table base */

typedef struct { 
  longword	pciphy;		/* 32-Bit Physical address */
  longword	mbzpad;		/* 32-Bit Must Be Zero pad */
} PCIADR;


/*******************************/
/* Driver Specific Definitions */
/*******************************/

#define	IC$K_UPDATE	2	/* Periodic update interval (Must be >= 2) */
#define	IC$K_TIMEOUTSEC	10	/* DMA timeout interval in seconds */
#define	IC$K_DEVBUFSIZ	512	/* Default buffer size (bytes) */

#define	IC$K_MAXBCNT		(1024*1024*1024)	/* Hardware maximum transfer length (bytes) */
#define	IC$K_UAMAXBCNT		65536			/* Maximum unaligned transfer length (bytes) */
#define	IC$K_DEVICE_IPL		21			/* Device interrupt priority level */
#define IC$K_FORK_IPL		8			/* Fork interrupt priority level */

#define	IC$K_REG0_SIZE		0x100			/* Number bytes to be mapped */
#define	IC$K_BAR0_CFG		0x10			/* Offset to BAR0 in config space */


/*************************************************/
/* Device Specific UCB/CRB Extension Definitions */
/*************************************************/

#define MAXIOBUF 100
typedef struct {
  CRCTX *crctx;
  longword pid;
  longword pciadr;
  longword bytes;
  longword usradr;
} DMACHANNEL;

typedef struct {
  DMACHANNEL dma[MAXIOBUF];
} UCBX;


/************************************/
/* Prototypes for External Routines */
/************************************/
int IOC$READ_PCI_CONFIG  (ADP *adp, int node, int off, int bcnt, int *data);
int IOC$WRITE_PCI_CONFIG (ADP *adp, int node, int off, int bcnt, int data);
#if (1!=1)
int IOC$MAP_IO   (ADP *adp, int node, uint64 *padr, int bcnt, int attr, uint64 *handle);
int IOC$UNMAP_IO (ADP *adp, uint64 *handle);
int IOC$READ_IO  (ADP *adp, uint64 *handle, int off, int bcnt, void *data);
int IOC$WRITE_IO (ADP *adp, uint64 *handle, int off, int bcnt, void *data);
#endif /* (1!=1) */


/******************************************/
/* Forward Prototypes for Driver Routines */
/******************************************/
longword DRIVER$INIT_TABLES ();							/* Init driver tables */
void     IC$STRUC_INIT   (CRB *crb, DDB *ddb, IDB *idb, ORB *orb, UCB *ucb);	/* Init driver structures on load */
void     IC$STRUC_REINIT (CRB *crb, DDB *ddb, IDB *idb, ORB *orb, UCB *ucb);	/* Reinit driver structures on reload */

longword IC$MAPDEVREG (IDB *idb, DDB *ddb, CRB *crb);				/* Map device registers routine */
longword IC$CTRL_INIT (IDB *idb, DDB *ddb, CRB *crb);				/* Controller init routine */
longword IC$UNIT_INIT (IDB *idb, UCB *ucb);					/* Unit init routine */

longword IC$FDT_NOP       (IRP *irp, PCB *pcb, UCB *ucb, CCB *ccb);		/* NOP FDT routine */
void     IC$STARTIO       (IRP *irp, UCB *ucb);					/* Start I/O routine */
void     IC$CANCELIO (longword chan, IRP *irp, PCB *pcb, UCB *ucb, longword reason); /* Cancel I/O routine */

longword IC$MAPIO       (IRP *irp, UCB *ucb, int function, int dmachan);					/* Start I/O routine */

/*
 * DRIVER$INIT_TABLES - Initialize Driver Tables routine
 *
 * Functional description:
 *
 *   This routine completes the initialization of the DPT, DDT, and FDT
 *   structures.  If a driver image contains a routine named DRIVER$INIT_TABLES
 *   then this routine is called once by the $LOAD_DRIVER service immediately
 *   after the driver image is loaded or reloaded and before any validity checks
 *   are performed on the DPT, DDT, and FDT.  A prototype version of these
 *   structures is built into this image at link time from the
 *   VMS$VOLATILE_PRIVATE_INTERFACES.OLB library.  Note that the device related
 *   data structures (e.g. DDB, UCB, etc.) have not yet been created when this
 *   routine is called.  Thus the actions of this routine must be confined to
 *   the initialization of the DPT, DDT, and FDT structures which are contained
 *   in the driver image.
 *
 * Calling convention:
 *
 *   status = DRIVER$INIT_TABLES();
 *
 * Input parameters:
 *
 *   None.
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   status     If the status is not successful, then the driver image will
 *              be unloaded.  Note that the ini_* macros used below will
 *              result in a return from this routine with an error status if
 *              an initialization error is detected.
 *
 * Implicit inputs:
 *
 *  driver$dpt, driver$ddt, driver$fdt
 *              These are the externally defined names for the prototype
 *              DPT, DDT, and FDT structures that are linked into this driver.
 *
 * Environment:
 * 
 *   Kernel mode, system context.
 */

longword DRIVER$INIT_TABLES()
{

/* Prototype driver DPT, DDT, and FDT will be pulled in from the  */
/* VMS$VOLATILE_PRIVATE_INTERFACES.OLB library at link time. */
 
    extern DPT driver$dpt;
    extern DDT driver$ddt;
    extern FDT driver$fdt;

    /************************************************************/
    /* Finish initialization of the Driver Prologue Table (DPT) */
    /************************************************************/
    ini_dpt_name        (&driver$dpt, "ICDRIVER");
    ini_dpt_adapt       (&driver$dpt, AT$_PCI);
    ini_dpt_flags       (&driver$dpt, DPT$M_SVP|DPT$M_SMPMOD|DPT$M_NOUNLOAD);
    ini_dpt_iohandles   (&driver$dpt, 1);
    ini_dpt_maxunits    (&driver$dpt, 1);
    ini_dpt_ucbsize     (&driver$dpt, sizeof(UCB)+sizeof(UCBX) );
    ini_dpt_struc_init  (&driver$dpt, IC$STRUC_INIT);
    ini_dpt_struc_reinit(&driver$dpt, IC$STRUC_REINIT);
    ini_dpt_end         (&driver$dpt);

    /************************************************************/
    /* Finish initialization of the Driver Dispatch Table (DDT) */
    /************************************************************/
    ini_ddt_csr_mapping	(&driver$ddt, (int (*)()) IC$MAPDEVREG);
    ini_ddt_ctrlinit	(&driver$ddt, (int (*)()) IC$CTRL_INIT);
    ini_ddt_unitinit	(&driver$ddt, (int (*)()) IC$UNIT_INIT);
    ini_ddt_start	(&driver$ddt, IC$STARTIO);
    ini_ddt_cancel	(&driver$ddt, IC$CANCELIO);
    ini_ddt_end		(&driver$ddt);

    /**************************************************************/
    /* Finish initialization of the Function Decision Table (FDT) */
    /**************************************************************/
    ini_fdt_act(&driver$fdt, IO$_NOP, (int (*)()) IC$FDT_NOP, DIRECT);

    ini_fdt_end(&driver$fdt);

    /* If we got this far, then everything worked, so return success. */

    return(SS$_NORMAL);

}



/*
 * IC$STRUC_INIT - Device Data Structure Initialization routine
 *
 * Functional description:
 *
 *   This routine is called once for each unit by the $LOAD_DRIVER service
 *   after that UCB is created.  At the point of this call the UCB has not
 *   yet been fully linked into the I/O database.  This routine is responsible
 *   for filling in driver specific fields that in the I/O database structures
 *   that are passed as parameters to this routine.
 *
 *   This routine is responsible for filling in the fields that are not
 *   affected by a RELOAD of the driver image.  In contrast, the structure
 *   reinitialization routine is responsible for filling in the fields that
 *   need to be corrected when (and if) this driver image is reloaded.
 *
 *   After this routine is called for a new unit, then the reinitialization
 *   routine is called as well.  Then the $LOAD_DRIVER service completes the
 *   integration of these device specific structures into the I/O database.
 *
 *   Note that this routine must confine its actions to filling in these I/O
 *   database structures and may not attempt to initialize the hardware device.
 *   Initialization of the hardware device is the responsibility of the 
 *   controller and unit initialization routines which are called some time
 *   later.
 *
 * Calling convention:
 *
 *   IC$STRUC_INIT(crb, ddb, idb, orb, ucb)
 *
 * Input parameters:
 *
 *   crb        Pointer to associated Controller Request Block.
 *   ddb        Pointer to associated Device Data Block.
 *   idb        Pointer to associated Interrupt Dispatch Block.
 *   orb        Pointer to associated Object Rights Block.
 *   ucb        Pointer to the Unit Control Block that is to be initialized.
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   None.
 *
 * Environment:
 * 
 *   Kernel mode, system context, IPL may be as high as 31 and may not be
 *   altered.
 *
 */ 

void IC$STRUC_INIT(CRB *crb, DDB *ddb, IDB *idb, ORB *orb, UCB *ucb)
{
  int i;
  UCBX 		*ucbx;			/* extended UCB */

  ucbx = (UCBX *)(ucb+1);		/* extended UCBX follows UCB */

  /* May have to use CRB for fork block */
  crb->crb$b_flck = SPL$C_IOLOCK8;   		/* Fork lock must be 8 */

  ucb->ucb$b_flck = SPL$C_IOLOCK8;	/* Fork lock must be 8 */
  ucb->ucb$b_dipl = IC$K_DEVICE_IPL;	/* Device IPL */
  ucb->ucb$l_devchar = 			/* Device Characteristics */
			DEV$M_AVL |	/*   Device is available */
			DEV$M_SHR |	/*   Device is sharable */
			DEV$M_IDV |	/*   Input device */
			DEV$M_ODV;	/*   Output device */
  ucb->ucb$b_devclass = DC$_MISC;	/* Device Class = Miscellaneous */
  ucb->ucb$b_devtype = 0;		/* Device Type = Unknown */
  ucb->ucb$w_devbufsiz = IC$K_DEVBUFSIZ; /* Default buffer size */
  ucb->ucb$l_ertcnt = 16;		/* Error retry count */
  ucb->ucb$l_ertmax = 16;		/* Maximum error retry count */

  for (i=0; i<MAXIOBUF; i++) {
    ucbx->dma[i].pciadr = 0;
    ucbx->dma[i].crctx = 0;
    ucbx->dma[i].pid = 0;
  }

  return;

}



/*
 * IC$STRUC_REINIT - Device Data Structure Re-Initialization routine
 *
 * Functional description:
 *
 *   This routine is called once for each unit by the $LOAD_DRIVER service
 *   immediately after the structure initialization routine is called.
 *
 *   Additionally, this routine is called once for each unit by the $LOAD_DRIVER
 *   service when a driver image is RELOADED.  Thus, this routine is
 *   responsible for filling in the fields in the I/O database structures
 *   that point into this driver image.
 *
 *   Note that this routine must confine its actions to filling in these I/O
 *   database structures.
 *
 * Calling convention:
 *
 *   IC$STRUC_REINIT(crb, ddb, idb, orb, ucb)
 *
 * Input parameters:
 *
 *   crb        Pointer to associated Controller Request Block.
 *   ddb        Pointer to associated Device Data Block.
 *   idb        Pointer to associated Interrupt Dispatch Block.
 *   orb        Pointer to associated Object Rights Block.
 *   ucb        Pointer to the Unit Control Block that is to be initialized.
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   None.
 *
 * Environment:
 * 
 *   Kernel mode, system context, IPL may be as high as 31 and may not be
 *   altered.
 *
 */ 

void IC$STRUC_REINIT(CRB *crb, DDB *ddb, IDB *idb, ORB *orb, UCB *ucb)
{
  extern DDT driver$ddt;

  ddb->ddb$ps_ddt = &driver$ddt;	/* Write DDT address into DDB */

  return;

}



/*
 * IC$MAPDEVREG - Map Device Registers into System Space Routine
 *
 * Functional description:
 *
 *   This routine is called by VMS before the controller initialization
 *   and unit initialization routines are called.  It runs at driver
 *   fork IPL, holding the fork lock, so any controller type (as opposed 
 *   to unit type) initialization that needs to be done in this context
 *   should be done here.  This routine reads the PCI configuration from
 *   the device hardware and maps the device registers into system space
 *   for access by the driver.  It also allocates a system buffer for
 *   use by the driver to communicate with the the device.  Finally, it
 *   maps the system buffer into PCI address space so that the device
 *   hardware can access it.
 *
 * Calling convention:
 *
 *   status = IC$MAPDEVREG(idb, ddb, crb)
 *
 * Input parameters:
 *
 *   idb        Pointer to Interrupt Data Block
 *   ddb	Pointer to Device Data Block
 *   crb	Pointer to Channel Request Block
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   None.
 *
 * Environment:
 * 
 *   Kernel mode, system context, fork IPL, fork lock held.
 */

longword IC$MAPDEVREG(IDB *idb, DDB *ddb, CRB *crb)
{
  PCIADR	pciadr;		/* PCI Address structure */
  longword	status;		/* Local function return status */

  status = SS$_NORMAL;			/* Assume success */

  /*************************/
  /* Map Runtime Registers */
  /*************************/
  if $SUCCESS(status) {	/* Read PCI config to get physical address */
    status = IOC$READ_PCI_CONFIG(
		idb->idb$ps_adp,	/* ADP address */
		crb->crb$l_node,	/* Node number */
		IC$K_BAR0_CFG,		/* PCI Config register offset */
		4,			/* Byte count */
		(int *)(&pciadr.pciphy) );	/* Address to store read data */
  }

  if $SUCCESS(status) {
    if ((pciadr.pciphy & 0X00000001) == 1) {	/* I/O Space Registers */

      pciadr.pciphy &= 0XFFFFFFFE;	/* Clear bit 0 */
      pciadr.mbzpad = 0;		/* Clear upper 32 bits */

      /* Map PCI I/O address to system virtual address */
      status = IOC$MAP_IO(	
		idb->idb$ps_adp,	/* ADP address */
		crb->crb$l_node,	/* Node number */
		(uint64 *)(&pciadr),	/* Bus physical address */
		IC$K_REG0_SIZE,		/* Number bytes to be mapped */
		IOC$K_BUS_IO_BYTE_GRAN,	/* I/O space attributes desired */
		&idb->idb$q_csr);	/* I/O Handle address */
    }
    else {	/* Memory Space Registers */

      pciadr.pciphy &= 0XFFFFFFF0;	/* Clear bits 0-3 */
      pciadr.mbzpad = 0;		/* Clear upper 32 bits */

      /* Map PCI memory address to system virtual address */
      status = IOC$MAP_IO(	
		idb->idb$ps_adp,	/* ADP address */
		crb->crb$l_node,	/* Node number */
		(uint64 *)(&pciadr),	/* Bus physical address */
		IC$K_REG0_SIZE,		/* Number bytes to be mapped */
		IOC$K_BUS_MEM_BYTE_GRAN,/* Memory space attributes desired */
		&idb->idb$q_csr);	/* I/O Handle address */
    }
  }

  return(status);

}



/*
 * IC$CTRL_INIT - Controller Initialization routine
 *
 * Functional description:
 *
 *   This routine is called once by the driver loading procecdure
 *   when processing a CONNECT command.  This routine is responsible
 *   for initializing the controller for a device.  For a device
 *   in which there is only one unit, the division of labor between
 *   the unit initialization routine and the controller initialization
 *   routines is not very important.  However, this routine is written
 *   correctly, since the driver can be compiled to support multiple
 *   units per controller, or a single unit per controller.
 *
 *   If the driver is compiled to support only one unit, this routine
 *   makes the only unit the permanent owner of the controller channel.
 *   It then resets the interface board, initializes the common
 *   memory, sends an initialization packet to the board, and
 *   enables interrupts on the PCI slot.
 *
 *   This routine is also called during power fail recovery.
 *
 * Calling convention:
 *
 *   status = IC$CTRL_INIT(idb, ddb, crb)
 *
 * Input parameters:
 *
 *   idb        Pointer to associated Interrupt Dispatch Block.
 *   ddb        Pointer to associated Device Data Block.
 *   crb        Pointer to associated Channel Request Block.
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   status     SS$_NORMAL indicates that the unit was initialized successfully.
 *              SS$_IVADDR indicates that an unexpected PCI I/O address or IRQ
 *                         level was detected.
 *
 * Environment:
 * 
 *   Kernel mode, system context, IPL$_POWER (=31).
 */

longword IC$CTRL_INIT(IDB *idb, DDB *ddb, CRB *crb)
{
  longword	status;		/* Local function return status */

  status = SS$_NORMAL;		/* Assume success */


  /************************************************/
  /* Make only unit permanent owner of controller */
  /************************************************/
  idb->idb$ps_owner = (UCB*) ddb->ddb$l_ucb;

  return(status);


}



/*
 * IC$UNIT_INIT - Unit Initialization routine
 *
 * Functional description:
 *
 *   This routine is called once for each unit by the $LOAD_DRIVER service
 *   after a new unit control block has been created, initialized, and
 *   fully integrated into the I/O database.
 *
 *   This routine is also called for each unit during power fail recovery.
 *
 *   This routine:
 *	Forks to map the device registers
 *	Maps the device registers
 *	Marks the device on line
 *
 * Calling convention:
 *
 *   status = IC$UNIT_INIT(idb, ucb)
 *
 * Input parameters:
 *
 *   idb        Pointer to associated Interrupt Dispatch Block.
 *   ucb        Pointer to the Unit Control Block that is to be initialized.
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   status     SS$_NORMAL indicates that the unit was initialized successfully.
 *
 *
 * Environment:
 * 
 *   Kernel mode, system context, IPL$_POWER (=31), primary CPU.
 */
 
longword IC$UNIT_INIT(IDB *idb, UCB *ucb)
{
  longword	status;		/* Local function return status */

  status = SS$_NORMAL;			/* Assume success */

  /****************************/
  /* Mark unit online/offline */
  /****************************/
  ucb->ucb$v_online = 1;

  return(status);

}



/*
 * IC$FDT_NOP - FDT Routine for NOP Function
 *
 * Functional description:
 *
 *   Sets return status to SS$_NORMAL and returns by
 *   jumping to EXE_STD$FINISHIO to finish I/O processing.
 *
 * Calling convention:
 *   status = IC$FDT_NOP(irp, pcb, ucb, ccb)
 *
 * Input parameters:
 *   irp        Pointer to I/O Request Packet
 *   pcb        Pointer to Process Control Block
 *   ucb        Pointer to Unit Control Block
 *   ccb        Pointer to Channel Control Block
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   status     SS$_FDT_COMPL
 *
 * Environment:
 * 
 *   Kernel mode, user process context, IPL$_ASTDEL (=2)
 */
#include "iceioctl.h"
#include "iceflex.h"

#define PIC_ADPADDR	ucb->ucb$ps_adp		/* address of ADP */
#define PIC_DEVNODE	crb->crb$l_node		/* device node */
#define PIC_IOHANDLE	&idb->idb$q_csr		/* device IO handle */
#define PIC_BUFADDR	irp->irp$l_qio_p1	/* user buffer addr */
#define PIC_BYTES	irp->irp$l_qio_p2	/* bytes to transfer */
#define PIC_FUNCTION	irp->irp$l_qio_p3	/* user function code */
#define PIC_OFFSET	irp->irp$l_qio_p4	/* register offset */
#define PIC_STATUS	irp->irp$l_qio_p5	/* bytes transferred */
#define PIC_DMACHAN	irp->irp$l_qio_p6	/* dma channel index (reuse) */


longword IC$FDT_NOP (IRP *irp, PCB *pcb, UCB *ucb, CCB *ccb)
{
  CRB		*crb;		/* Pointer to Channel Request Block */
  VEC		*vec;		/* Pointer to Vector Block */
  IDB		*idb;		/* Pointer to Data Block */
  longword	status,stat;	/* Local function return status */
  UCBX		*ucbx;		/* extended UCBX */
  int		chan;		/* Current dma channel */
  int		i,j,k;		/* Local counters */

  unsigned char *bdata;		/* byte data reference */
  int		worda,wordb;	/* load IOC temps */

  crb = ucb->ucb$l_crb;		/* crb := address of CRB */
  vec = (VEC*)crb->crb$l_intd;	/* vec := address of VEC */
  idb = vec->vec$l_idb; 	/* idb := address of IDB */
  ucbx = (UCBX *)(ucb+1);	/* extended UCBX follows UCB */

  i = PIC_BYTES;		/* normal|default completion status */
  status = SS$_NORMAL;		/* assume success */

  /* Validate user buffer alignment */
  if ( (PIC_BUFADDR % 4) != 0 ) status = SS$_BUFNOTALIGN;
  else  status = SS$_NORMAL; 

  /* Validate user buffer size */
  if ( PIC_BYTES < 0  || PIC_BYTES > IC$K_MAXBCNT || 
        (PIC_BYTES & 0x3) != 0 ) status = SS$_IVBUFLEN;
  if (!(status&1)) goto FINISH;

  /* Lock user buffer in memory */
  if (PIC_BYTES>0) status = EXE_STD$MODIFYLOCK (irp,pcb,ucb,ccb,
		(void*)PIC_BUFADDR, PIC_BYTES, NULL);
  if (!(status&1)) return(status);	/* EXE$ABORT already called */

  switch (PIC_FUNCTION) {

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_MCSR, 4, (void *)(&stat));
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
	else break;
        while (j>0) { 
	  status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_FIFO, 4, (void *)(PIC_BUFADDR+i) );  
	  i+=4; j--;
        }
      }
    } 
    else if ((PIC_OFFSET&0XFFFFF000) != 0)  /* valid register offset ? */
      status = SS$_BADPARAM;
    else 
      status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			PIC_OFFSET, PIC_BYTES, (void *)PIC_BUFADDR );		
    break;

  case IOCTL_WRITE:  			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
        status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE,
                        REG_MCSR, 4, (void *)(&stat));
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6; 
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4; 
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2; 
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
	else break;
        while (j>0) {
	  status = IOC$WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE,
                        REG_FIFO, 4, (void *)(PIC_BUFADDR+i) );
	  i+=4; j--;
        }
      }
    }
    else if ((PIC_OFFSET&0XFFFFF000) != 0)  /* valid register offset ? */
      status = SS$_BADPARAM;
    else 
      status = IOC$WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			PIC_OFFSET, PIC_BYTES, (void *)PIC_BUFADDR );		
    break;

  case IOCTL_ACMD:			/* command to SHARC processor */

    for (j=16; j>=0; j-=4) 
      status = IOC$WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_MBX+j, 4, (void *)(PIC_BUFADDR+j) );
    for (i=0,j=0; j<32; j++) {    /* cant wait forever in kernel mode */
      status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_MCSR, 4, (void *)(&stat) );
      if ( (status == SS$_NORMAL) && ((stat&MCSR_IMB1F) != 0x0) ) {
        status = IOC$READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_IMB1, 4, (void *)PIC_BUFADDR );
	i=4; break;
      }
    }
    break;

  case IOCTL_MAP:			/* map user buffer to PCI */

    /* find an empty IOBUF index */
    for (chan=0; chan<MAXIOBUF && ucbx->dma[chan].pciadr!=0; chan++);
    if (chan>=MAXIOBUF) status = SS$_ABORT;
    if (!(status&1)) break;
    PIC_DMACHAN = chan;

    /* jump to system context, fork IPL to map buffer */
    EXE_STD$QIODRVPKT(irp, ucb); 

    return(SS$_NORMAL);
    break;

  case IOCTL_UNMAP:			/* unmap user buffer from PCI */

    for (chan=0; chan<MAXIOBUF && ucbx->dma[chan].pciadr!=PIC_OFFSET; chan++);
    if (chan>=MAXIOBUF) status = SS$_ABORT;
    if (!(status&1)) break;
    PIC_DMACHAN = chan;

    /* jump to system context, fork IPL to unmap buffer */
    EXE_STD$QIODRVPKT(irp, ucb);

    return(SS$_NORMAL); 
    break;

  case IOCTL_QMAP:			/* query of user buffer map */

    for (i=0,chan=0; chan<MAXIOBUF && i==0; chan++) {
      if (PIC_OFFSET >= (ucbx->dma[chan].pciadr) &&
          PIC_OFFSET <= (ucbx->dma[chan].pciadr+ucbx->dma[chan].bytes)) 
    		i = ucbx->dma[chan].bytes; 
    }
    break;

  case IOCTL_LIOC:			/* load IOC words */

    bdata = (unsigned char *)PIC_BUFADDR;
    for (k=0; k<PIC_BYTES; k++) {
      worda = (int)bdata[k];
      for (j=0; j<8; j++) {
	wordb = worda>>j;
        status = IOC$WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_IOC, 4, (void *)(&wordb) );
      }
    }
    break;


  case IOCTL_VERSION:			/* get driver version */

    *(int*)PIC_BUFADDR = ICE_DRIVER_VERSION;
    break;

  default:				/* handle unknown function codes */
    status = SS$_BADPARAM;

  }

  /* fill in IO status block and signal completion */
  FINISH:
  irp->irp$l_iost1 = status;
  irp->irp$l_iost2 = i;
  status = EXE_STD$FINISHIO (irp,ucb);
  return(status);

}



/*
 * IC$STARTIO - Start I/O routine
 *
 * Functional description:
 *
 *   This routine is the driver start I/O routine.  This routine is
 *   called by EXE_STD$QIODRVPKT (from the FDT routines) to process the next
 *   I/O request that has been sent to this device by the SYS$QIO or 
 *   SYS$QIOW system services.
 *
 *   Before this routine is called, ucb$v_cancel, ucb$v_int, ucb$v_tim, and
 *   ucb$v_timout are cleared.  The ucb$l_svapte, ucb$l_boff, and ucb$l_bcnt
 *   cells are set from their corresponding IRP cells.  However, unlike their
 *   IRP counterparts, these UCB cells are working storage and can be changed
 *   by a driver.
 *
 * Calling convention:
 *
 *   IC$STARTIO(irp, ucb)
 *
 * Input parameters:
 *
 *   irp        Pointer to I/O Request Packet
 *   ucb        Pointer to Unit Control Block
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   None.
 *
 * Environment:
 * 
 *   Kernel mode, system context, fork IPL, fork lock held.
 */

void IC$STARTIO(IRP *irp, UCB *ucb)
{
  int 	status;
  UCBX	*ucbx = (UCBX *)(ucb+1);	/* extended UCBX follows UCB */

  status = IC$MAPIO (irp, ucb, PIC_FUNCTION, PIC_DMACHAN);

  /* complete I/O processing - return PCI DMA ADDR in status */
  IOC_STD$REQCOM(status, ucbx->dma[PIC_DMACHAN].pciadr, ucb);	

}

longword IC$MAPIO(IRP *irp, UCB *ucb, int function, int chan)
{
  longword	status;		/* Function call status */
  CRCTX		*crctx;		/* Pointer to CRCTX */
  longword	bytes,pages;
  CRB		*crb;		/* Pointer to Channel Request Block */
  VEC		*vec;		/* Pointer to Vector Block */
  IDB		*idb;		/* Pointer to Data Block */
  UCBX		*ucbx;		/* extended UCB */
  longword	fstatus;	/* final cleanup status */
  DMACHANNEL	*dma;		/* dma channel block */

  crb = ucb->ucb$l_crb;		/* crb := address of CRB */
  vec = (VEC*)crb->crb$l_intd;	/* vec := address of VEC */
  idb = vec->vec$l_idb; 	/* idb := address of IDB */
  ucbx = (UCBX *)(ucb+1);	/* extended UCBX follows UCB */

  dma = ucbx->dma + chan;	/* dma channel header */

  switch (function) {

  case IOCTL_MAP:		/* Map the (user's) DMA buffer */

    /* allocate a resource context block */
    status = IOC$ALLOC_CRCTX(
	idb->idb$ps_adp->adp$l_crab,	/* CRAB address */
	&crctx,				/* Address to store allocated CRCTX */
	ucb->ucb$b_flck);		/* Index of fork lock */
    if (!(status&1)) goto BAD_ALLOC_CRCTX;

    /* Compute Number of Adapter Mapping Registers Required */
    bytes = irp->irp$l_boff +	/* Byte offset in page */
          irp->irp$l_bcnt +	/* Byte count */
	  MMG$GL_BWP_MASK +	/* Round up to next full page */
	  MMG$GL_PAGE_SIZE +	/* Add extra "no access page" */
	  MMG$GL_PAGE_SIZE;	/* Add second guard page */
    pages = (bytes >> ABS(MMG$GL_VA_TO_VPN));	/* Convert bytes to pages */

    /* Initialize CRCTX */
    crctx->crctx$l_item_cnt = pages;	/* Load CRCTX item count */
    crctx->crctx$l_up_bound = 0;	/* Clear upper bound */
    crctx->crctx$l_low_bound = 0;	/* Clear low bound */
    crctx->crctx$l_flags = 0;		/* Clear flags */
    crctx->crctx$l_callback = 0;	/* Clear callback */

    /* Allocate Adapter Mapping Registers */
    status = IOC$ALLOC_CNT_RES (
	ucb->ucb$ps_adp->adp$l_crab,	/* Address of CRAB */
	crctx, 0,0,0);			/* Address of CRCTX */
    if (!(status&1)) goto BAD_ALLOC_CNT_RES;

    /* Load Adapter Mapping Registers */
    status = IOC$LOAD_MAP(
	ucb->ucb$ps_adp,
	crctx,
	ucb->ucb$l_svapte,
	ucb->ucb$l_boff,
	(void *)(&dma->pciadr) );
    if (!(status&1)) goto BAD_LOAD_MAP;

    /* save context for subsequent unmap */
    dma->pid = irp->irp$l_pid;
    dma->bytes = irp->irp$l_bcnt;
    dma->crctx = crctx;

    break;

  case IOCTL_UNMAP:		/* UnMap the (user's) DMA buffer */

    crctx = dma->crctx;
    status = SS$_NORMAL;		/* assume success */

    BAD_LOAD_MAP:
    BAD_ALLOC_CNT_RES:
    if (crctx == 0) goto NO_CRCTX;
    fstatus = IOC$DEALLOC_CNT_RES(
	ucb->ucb$ps_adp->adp$l_crab,	/* Address of CRAB */
	crctx);				/* Address of CRCTX */

    BAD_ALLOC_CRCTX:
    if (crctx == 0) goto NO_CRCTX;
    fstatus = IOC$DEALLOC_CRCTX(crctx);

    NO_CRCTX:
    dma->crctx = 0;
    dma->pciadr = 0;
    dma->bytes = 0;
    dma->pid = 0;

    break;

  default:
    status = SS$_BADPARAM;

  }

  return (status);
}



/*
 * IC$CANCELIO - Cancel Pending I/O on Device routine
 *
 * Functional description:
 *
 *
 * Calling convention:
 *
 *   IC$CANCEL(channel,irp,pcb,ucb,reason)
 *
 * Input parameters:
 *
 *   channel	Channel index number 
 *   irp        Pointer to I/O Request Packet
 *   pcb	Pointer to Process Control Block of requesting process
 *   ucb        Pointer to Unit Control Block
 *   reason	Reason for cancellation:
 *		(=0) CAN$C_CANCEL	Invoked due to $CANCEL service
 *		(=1) CAN$C_DASSGN	Invoked due to $DASSGN service
 *		(=2) CAN$C_AMBXDGN	Invoked due to Mailbox disassociation
 *		(=3) CAN$C_MSCPSERVER	Invoked by MSCP server due to 
 *					   connection loss with client node
 *
 * Output parameters:
 *
 *   None.
 *
 * Return value:
 *
 *   None.
 *
 * Environment:
 * 
 *   Kernel mode, user context, fork IPL, fork lock held
 *
 */

void IC$CANCELIO(longword channel, IRP *irp, PCB *pcb, UCB *ucb, longword reason)
{
  longword	status;		/* Local function return status */
  CRB		*crb;		/* Pointer to Channel Request Block */
  VEC		*vec;		/* Pointer to Vector Block */
  IDB		*idb;		/* Pointer to Data Block */
  UCBX		*ucbx;		/* extended UCB */
  int		chan;		/* dma channel */
  int		reset;		/* register value to reset SHARC */

  crb = ucb->ucb$l_crb;		/* crb := address of CRB */
  vec = (VEC*)crb->crb$l_intd;	/* vec := address of VEC */
  idb = vec->vec$l_idb; 	/* idb := address of IDB */
  ucbx = (UCBX *)(ucb+1);	/* extended UCBX follows UCB */
  reset = MCSR_RESET;

  ioc_std$cancelio(channel,irp,pcb,ucb);

  if (reason == 1) {	/* deassgn - IRP is not valid */

    for (chan=0; chan<MAXIOBUF; chan++) {

      if (ucbx->dma[chan].pid==pcb->pcb$l_pid && ucbx->dma[chan].pciadr!=0) {

        /* unexpected termination - reset card to avoid stray DMAs */
	status = IOC$WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_MCSR, 4, (void *)(&reset) );		

	/* deallocate/UnMap the stray buffer */
	status = IC$MAPIO(irp, ucb, IOCTL_UNMAP, chan);
      }
    }
  }
}


