/* ---------- P C I C O N F I G ---------- */

/*
_Description:	This program walks the adapter list
		in the I/O database looking for ICE-PIC cards.
		When an ICE-PIC is found, the program adds its 
		TR number, Node number, and Interrupt Vector
		to a list.  These three lists are used to
		define local symbols which are used by the 
		BUILD.COM command file to construct the
		CONNECT.COM command file.

_Inputs:	none
		
_Switches:	

_Outputs:	The following global symbols are defined:

		devtrs   - Contains list of TR numbers for the devices	
		devnodes - Contains list of NODE numbers for the devices
		devvecs  - Contains list of Interrupt Vectors for the devices


_Author:	John R. Hill, Jeff Schoen

_Date:		01-DEC-1996

_Note:		Special thanks to Doug Smith at DEC.

*/


#include	<ints.h>
#include	<stdio.h>	/* Standard I/O definitions */
#include	<string.h>	/* String function definitions */
#include	<ctype.h>	/* Type definitions */
#include	<builtins.h>	/* Built in functions */
#include	<libclidef.h>	/* LIB CLI definitions */
#include	<ssdef.h>	/* System service definitions */
#include	<descrip.h>	/* Data descriptor definitions */
#include	<iodef.h>	/* QIO function definitions */

#include	<adpdef.h>	/* I/O Adapter definitions */
#include	<hwrpbdef.h>	/* ??? definitions */
#include	<busarraydef.h>	/* Adapter bus array definitions */
#include	<dcdef.h>	/* Device & adapter type definitions */
#include	<vecdef.h>	/* Interrupt vector definitions */
#include	<crbdef.h>	/* CRB definitions */
#include	<iocdef.h>	/* I/O Controller definitions */


#define	$failure( a ) ( !(a & 1) )
#define $success( a ) ( a & 1 )

#define	true	(1==1)
#define	false	(1!=1)

#define	DEBUG	false

#define	PCIDEVICE	0X77771172	/* AMCC PCI I/F chip */


typedef	unsigned long	longword;
typedef unsigned short	word;
typedef	unsigned char	byte;


/*************************/
/* Structure Definitions */
/*************************/

typedef struct _BusArrayHeader
{
  ADP		*busarray$ps_parent_adapter;	/* Address of ADP */
  longword	busarray$l_fill1;		/* Fill longword */
  word		busarray$w_size;		/* Structure size (bytes) */
  byte		busarray$b_type;		/* Structure type */
  byte		busarray$b_subtype;		/* Structure subtype */
  longword	busarray$l_bus_type;		/* Bus Type code */
  longword	busarray$l_bus_node_cnt;	/* Number of entries in Bus Array */
  longword	busarray$l_fill2;		/* Fill to quadword boundary */

} BUSARRAYHEADER;

/* Note:	BUSARRAYHEADER is defined in <busarraydef.h> but
		it includes the start of the array at the end,
		so it does not have the correct size and hence
		cannot be used to build a bus array structure in
		a simple manner.
*/

#define	MAX$BUSARRAYENTRY	32

typedef struct _BusArray
{
  BUSARRAYHEADER	header;			/* Bus array header */
  BUSARRAYENTRY		entry[MAX$BUSARRAYENTRY]; /* Bus array entries */

} BUSARRAY;



/*******************************/
/* Forware Function Prototypes */
/*******************************/

void GetADP();
void GetBusArray();
void PrintQuad(int64 n);


/*********************/
/* External Routines */
/*********************/

extern void	SYS$CMKRNL();		/* Change mode to kernel */
extern longword	LIB$SET_SYMBOL();	/* Set symbol value */

extern void 	exit();			/* Exit with completion status */


/*****************************/
/* External Global Variables */
/*****************************/

extern ADP	*IOC$GL_ADPLIST;
extern HWRPB	*EXE$GPL_HWRPB_L;


/********************/
/* Global Variables */
/********************/

ADP		l_ADP;		/* Local copy of ADP */
ADP		*ADP_addr;	/* Address of ADP */

BUSARRAY	l_BUSARRAY;	/* Local copy of BUSARRAY */
BUSARRAY	*BUSARRAY_addr;	/* Address of BUSARRAY */

$DESCRIPTOR(trsdsc,"devtrs");
$DESCRIPTOR(noddsc,"devnodes");
$DESCRIPTOR(vecdsc,"devvecs");


#define	NUM_ARGS	(0)


main( argc, argv )
int	argc;
char	*argv[];
{
	longword	status;		/* function return status */

	int		largc;		/* Local copy of argc */
	char		**largv;	/* Local copy of argv */
	char		*swp;		/* Switch pointer */

	longword	show_adplist;	/* Show adapter list */
	longword	show_busarray;	/* Show bus array for PCI adapters */
	longword	show_symbols;	/* Show symbol definitions (output) */
	longword	show_devinfo;	/* Show all info about PCI Device interface */


	FILE		*ofp;		/* Output file pointer */
	longword	npcidev;	/* Number of PCI device units */

	char		trs[132];	/* String of TR numbers */
	char		nodes[132];	/* String of NODE numbers */
	char		vecs[132];	/* String of VECTOR numbers */

	struct dsc$descriptor valdsc;	/* Descriptor for symbol value */
	longword	symtbl;		/* Symbol table type */

	char		stmp[132];	/* Temporary string */
	int		i,j,k;		/* Temporary integers */
	longword	*lp;		/* Temporary longword pointer */



	/**************/
	/* Initialize */
	/**************/
	show_adplist = false;	/* Show each adapter in list */
	show_busarray = false;	/* Show bus array for PCI adapters */
	show_symbols = false;	/* Show symbol definitions (output) */
	show_devinfo = false;	/* Show info about SAM interface */

	npcidev = 0;	/* Initialize PCI Device counter */
	trs[0] = '\0';
	nodes[0] = '\0';
	vecs[0] = '\0';

	/******************************/
	/* Get Arguments and Switches */
	/******************************/
	largc = argc;
	largv = argv;

	/***************************/
	/* Optional Switches First */
	/***************************/
	while ((--largc > 0) && (*++largv)[0] == '-')
	{ for (swp = largv[0]+1;*swp != '\000';swp++)
	  { switch(*swp) {
	    case 'a':		/* Show entire ADP configuration */
	    case 'A':
	      show_adplist = true;
	      break;

	    case 'b':
	    case 'B':
	      show_busarray = true;
	      break;

	    case 'i':
	    case 'I':
	      show_devinfo = true;
	      break;

	    case 's':
	    case 'S':
	      show_symbols = true;
	      break;

	    default:
	      fprintf(stderr,"\npciconfig: illegal option: %c\n\n",*swp);
	    } /* endswitch:*swp */
	  } /* endfor:swp */
	} /* endwhile:largc */

	/*************/
	/* Arguments */
	/*************/
	if (largc != NUM_ARGS)
	{ fprintf(stderr,"\nUsage: pciconfig [-abis]\n\n");
	  exit(1);
	} /* endif:largc */





	if (show_adplist || show_busarray || show_devinfo || show_symbols)
	  printf("\n");

	ADP_addr = IOC$GL_ADPLIST;	/* Point to first I/O Adapter */

	if (show_adplist)
	{ printf("IOC$GL_ADPLIST: %08X (hex)\n",ADP_addr);
	  printf("\n");
	} /* endif:show_adplist */


	while (true)
	{
	  SYS$CMKRNL(&GetADP,0);
	  if (show_adplist)
	  { printf("        ADP Address: %08X (hex)\n",ADP_addr);
	    printf("l_ADP.adp$l_adptype: %08X (hex)  ",
		l_ADP.adp$l_adptype);
	    switch (l_ADP.adp$l_adptype) {
	    case AT$_MBA: printf("(MASSBUS)"); break;
	    case AT$_UBA: printf("(UNIBUS)"); break;
	    case AT$_DR: printf("(DR32)"); break;
	    case AT$_MPM: printf("(Multi-Port)"); break;
	    case AT$_CI: printf("(CI)"); break;
	    case AT$_NULL: printf("(NULL)"); break;
	    case AT$_BDA: printf("(BI)"); break;
	    case AT$_DMB32: printf("(DMB32)"); break;
	    case AT$_DRB32: printf("(DRB32)"); break;
	    case AT$_BVP: printf("(BVP)"); break;
	    case AT$_BVP_SSP: printf("(BVP Storage System)"); break;
	    case AT$_BVP_NIP: printf("(BVP NI Port)"); break;
	    case AT$_KA410: printf("(VAXstar System)"); break;
/*	    case AT$_KA420: printf("(PVAX System)"); break;	*/
	    case AT$_GENBI: printf("(Generic BI)"); break;
	    case AT$_NBI: printf("(NBI)"); break;
	    case AT$_DISK9: printf("(DISK9)"); break;
	    case AT$_XBI: printf("(XBI)"); break;
	    case AT$_TERM9: printf("(TERM9)"); break;
	    case AT$_TAPE9: printf("(TAPE9)"); break;
	    case AT$_PRTR9: printf("(PRTR9)"); break;
	    case AT$_SFUN9: printf("(SFUN9)"); break;
	    case AT$_USER9: printf("(USER9)"); break;
	    case AT$_MBUSIO: printf("(MBUSIO)"); break;
	    case AT$_MBUSGFX: printf("(MBUSGFX)"); break;
	    case AT$_KA640: printf("(KA640)"); break;
	    case AT$_XWATCH: printf("(XWATCH)"); break;
	    case AT$_XBI_PLUS_XMI: printf("(XBI Plus XMI)"); break;
	    case AT$_XBI_PLUS_BI: printf("(XBI Plus BI)"); break;
	    case AT$_XJA: printf("(XJA)"); break;
	    case AT$_HSX50: printf("(HSX50)"); break;
/*	    case AT$_KDM70: printf("(KDM70)"); break;	*/
	    case AT$_NI: printf("(NI)"); break;
	    case AT$_KA43: printf("(KA43)"); break;
	    case AT$_SJA: printf("(SJA)"); break;
	    case AT$_GENXMI: printf("(GENXMI)"); break;
	    case AT$_KA440: printf("(KA440)"); break;
/*	    case AT$_KA46: printf("(KA46)"); break;	*/
	    case AT$_KA520: printf("(KA520)"); break;
	    case AT$_XSA: printf("(XSA)"); break;
	    case AT$_XZA: printf("(XZA)"); break;
/*	    case AT$_XZA_SCSI: printf("(XZA_SCSI)"); break;	*/
	    case AT$_VME: printf("(VME)"); break;
	    case AT$_IOP: printf("(IOP)"); break;
	    case AT$_LAMB: printf("(LAMB)"); break;
	    case AT$_KA49: printf("(KA49)"); break;
	    case AT$_TC: printf("(TC)"); break;
	    case AT$_X1303: printf("(X1303)"); break;
	    case AT$_XMI: printf("(XMI)"); break;
	    case AT$_FBUS: printf("(FBUS)"); break;
	    case AT$_COREIO: printf("(COREIO)"); break;
	    case AT$_KA0202: printf("(KA0202)"); break;
	    case AT$_KA0202_LBUS: printf("(KA0202_LBUS)"); break;
	    case AT$_KA0302: printf("(KA0302)"); break;
	    case AT$_KA0402: printf("(KA0401)"); break;
	    case AT$_TURBO_SCSI: printf("(Turbo SCSI)"); break;
	    case AT$_CIMNA: printf("(CIMNA)"); break;
	    case AT$_XZA_DSSI: printf("(XZA_DSSI)"); break;
	    case AT$_DEMNA: printf("(DEMNA)"); break;
	    case AT$_FFA: printf("(FFA)"); break;
	    case AT$_KA0602: printf("(KA0602)"); break;
	    case AT$_EISA: printf("(EISA)"); break;
	    case AT$_VTI_COMBO: printf("(VTI Combo)"); break;
	    case AT$_AHA_1742A: printf("(AHA 1742A)"); break;
	    case AT$_KA0902: printf("(KA0902)"); break;
	    case AT$_PCI: printf("(PCI)"); break;
	    case AT$_KA0802: printf("(KA0802)"); break;
	    case AT$_MULTIFUNCTION_PCI: printf("(Multifunction PCI)"); break;
	    case AT$_ISA: printf("(ISA)"); break;
	    case AT$_XBUS: printf("(XBUS)"); break;
	    case AT$_KA0C05: printf("(KA0C05)"); break;
	    case AT$_KA0E04: printf("(KA0E04)"); break;
	    case AT$_KA0D02: printf("(KA0D02)"); break;
	    case AT$_THIRDPARTY0: printf("(THIRDPARTY0)"); break;
	    case AT$_THIRDPARTY1: printf("(THIRDPARTY1)"); break;
	    case AT$_THIRDPARTY2: printf("(THIRDPARTY2)"); break;
	    case AT$_THIRDPARTY3: printf("(THIRDPARTY3)"); break;
	    case AT$_THIRDPARTY4: printf("(THIRDPARTY4)"); break;
	    case AT$_THIRDPARTY5: printf("(THIRDPARTY5)"); break;
	    case AT$_THIRDPARTY6: printf("(THIRDPARTY6)"); break;
	    case AT$_THIRDPARTY7: printf("(THIRDPARTY7)"); break;
	    case AT$_MULTIFUNCTION_ISA: printf("(MUltifunction ISA)"); break;
	    case AT$_KA0F05: printf("(KA0F05)"); break;
	    case AT$_LMCP: printf("(LMCP)"); break;
	    case AT$_TIOP: printf("(TIOP)"); break;
	    case AT$_ITIOP: printf("(ITIOP)"); break;
	    case AT$_KA1102: printf("(KA1102)"); break;
	    case AT$_KA1504: printf("(KA1504)"); break;
	    case AT$_HPC: printf("(HPC)"); break;
	    case AT$_PCMCIA: printf("(PCMCIA)"); break;
	    case AT$_KA1402: printf("(KA1402)"); break;
	    case AT$_KA0905: printf("(KA0905)"); break;
	    default:      printf("Unknown"); break;
	    } /* endswitch:adptype */
	    printf("\n");
	    printf("     l_ADP.adp$l_tr: %08X (hex)   %d (dec)\n",
		l_ADP.adp$l_tr,l_ADP.adp$l_tr);
	    printf("   l_ADP.adp$l_link: %08X (hex)\n",
		l_ADP.adp$l_link);
	    printf("\n");
	  } /* endif:show_adplist */

	  switch (l_ADP.adp$l_adptype) {

	  case AT$_PCI:
	    if (show_busarray)
	    { printf("Found PCI adapter\n");
	    } /* endif:show_adplist */


	    BUSARRAY_addr = (BUSARRAY *)l_ADP.adp$ps_bus_array;
	    SYS$CMKRNL(GetBusArray,0);
	    for (i=0;i<l_BUSARRAY.header.busarray$l_bus_node_cnt;i++)
	    {
	      if (show_busarray)
	      { printf("entry[%02d].busarray$q_hw_id: %016LX (hex)\n",
			i,l_BUSARRAY.entry[i].busarray$q_hw_id);
	      } /* endif:show_busarray */

	      if (l_BUSARRAY.entry[i].busarray$q_hw_id == PCIDEVICE)
	      {
	        if (show_devinfo)
	        { printf("Found PCI Device\n");
	          printf("            ADP Address:         %08X (hex)\n",ADP_addr);
	          printf("    l_ADP.adp$l_adptype:         %08X (hex)  (PCI)\n",
			l_ADP.adp$l_adptype);
	          printf("         l_ADP.adp$l_tr:         %08X (hex)\n",
			l_ADP.adp$l_tr);
	          printf("  busarray entry offset:         %08X (hex)   %d (dec)\n",
			i,i);
	          printf("       busarray$q_hw_id: %016LX (hex)\n",
			l_BUSARRAY.entry[i].busarray$q_hw_id);
	          printf("         busarray$q_csr: %016LX (hex)\n",
			l_BUSARRAY.entry[i].busarray$q_csr);
	          printf(" busarray$l_node_number:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$l_node_number);
	          printf("       busarray$l_flags:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$l_flags);
	          printf("       *busarray$ps_crb:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$ps_crb);
	          printf("       *busarray$ps_adp:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$ps_adp);
	          printf("  busarray$l_autoconfig:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$l_autoconfig);
	          printf("     busarray$l_ctrlltr:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$l_ctrlltr);
	          printf("busarray$q_bus_specific: %016LX (hex)\n",
			l_BUSARRAY.entry[i].busarray$q_bus_specific);
	          printf("              TR number:         %08X (hex)   %d (dec)\n",
			l_ADP.adp$l_tr,l_ADP.adp$l_tr);
	          printf("            Node number:         %08X (hex)   %d (dec)\n",
			l_BUSARRAY.entry[i].busarray$l_node_number,
			l_BUSARRAY.entry[i].busarray$l_node_number);
	          printf("       Interrupt vector:         %08X (hex)\n",
			l_BUSARRAY.entry[i].busarray$l_bus_specific_l);
	          printf("\n");
	        } /* endif:show_devinfo */

		npcidev += 1;	/* Bump PCI Device count */

		sprintf(stmp,"%04X,",
		    l_ADP.adp$l_tr);
		strcat(trs,stmp);

		sprintf(stmp,"%04X,",
		    l_BUSARRAY.entry[i].busarray$l_node_number);
		strcat(nodes,stmp);

		sprintf(stmp,"%04X,",
		    l_BUSARRAY.entry[i].busarray$l_bus_specific_l);
		strcat(vecs,stmp);

	      } /* endif:hw_id */

	    } /* endfor:i */

	    if (show_busarray)
	      printf("\n");

	    break;

	  default:
	    ;

	  } /* endswitch:adp$l_adptype */



	  if (l_ADP.adp$l_link == NULL)		/* Exit Loop */
	    break;



	  ADP_addr = l_ADP.adp$l_link;

	} /* endwhile:true */

	if (show_symbols)
	{ printf("Local symbol   DEVTRS >%s<\n",trs);
	  printf("Local symbol DEVNODES >%s<\n",nodes);
	  printf("Local symbol  DEVVECS >%s<\n",vecs);
	  printf("\n");
	} /* endif:show_symbols */


	/******************/
	/* Define Symbols */
	/******************/

	symtbl = LIB$K_CLI_LOCAL_SYM;

	valdsc.dsc$w_length = strlen(trs);
	valdsc.dsc$b_dtype = DSC$K_DTYPE_T;
	valdsc.dsc$b_class = DSC$K_CLASS_D;
	valdsc.dsc$a_pointer = trs;
	status = LIB$SET_SYMBOL(&trsdsc,&valdsc,&symtbl);
	if $failure(status)
	{ printf("*** Error *** Setting local symbol: %s\n",
		trsdsc.dsc$a_pointer);
	  exit(status);
	} /* endif:status */

	valdsc.dsc$w_length = strlen(nodes);
	valdsc.dsc$b_dtype = DSC$K_DTYPE_T;
	valdsc.dsc$b_class = DSC$K_CLASS_D;
	valdsc.dsc$a_pointer = nodes;
	status = LIB$SET_SYMBOL(&noddsc,&valdsc,&symtbl);
	if $failure(status)
	{ printf("*** Error *** Setting local symbol: %s\n",
		trsdsc.dsc$a_pointer);
	  exit(status);
	} /* endif:status */

	valdsc.dsc$w_length = strlen(vecs);
	valdsc.dsc$b_dtype = DSC$K_DTYPE_T;
	valdsc.dsc$b_class = DSC$K_CLASS_D;
	valdsc.dsc$a_pointer = vecs;
	status = LIB$SET_SYMBOL(&vecdsc,&valdsc,&symtbl);
	if $failure(status)
	{ printf("*** Error *** Setting local symbol: %s\n",
		trsdsc.dsc$a_pointer);
	  exit(status);
	} /* endif:status */



	exit(1);

} /*** endmain ***/



#if false
	printf(" sizeof(BUSARRAYHEADER): %d (dec)\n",
		sizeof(BUSARRAY_HEADER));

	printf("  sizeof(BUSARRAYENTRY): %d (dec)\n",
		sizeof(BUSARRAYENTRY));
#endif /* false */



void GetADP()
{
  memcpy(&l_ADP,ADP_addr,sizeof(ADP));

} /*** endprocedure GetADP ***/



void GetBusArray()
{
  memcpy(&l_BUSARRAY,BUSARRAY_addr,sizeof(BUSARRAY));

} /*** endprocedure GetBusArray***/



#if false
void PrintQuad(int64 n)
{
  char str1[20],str2[20];

  sprintf(str1,"%016LX",n);
  strncpy(str2,str1,8);
  str2[8] = ',';
  strncpy(&str2[9],&str1[8],8);
  printf("%s",str2);

} /*** endprocedure PrintQuad ***/
#endif /* false */
