/*
  Verilog modules for simulating various Xilinx primitives
  This also allows similar primitives to be inferred on Altera devices

  Note: In verilator, counters must be in their own always block

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

*/
`include "../lib/ramdefs.h"

module virtex();

endmodule

module BUF (O,I);
  input I;
  output O;
  assign O = I;
endmodule

module BUFE (O, EN, I);
  output O;
  input EN,I;
  assign O = I && EN;
endmodule

module BUFG (O, I);
  output O;
  input I;
  assign O = I;
endmodule

module BUFGCE (O, CE, I);
  output O;
  input CE,I;
  assign O = CE && I;
endmodule

module BUFGMUX (O,I1,I2,S);
  input I1,I2,S;
  output O;
  assign O = S? I2:I1;
endmodule

module BUFG_GT (O, I);
  output O;
  input I;
  assign O = I;
endmodule

module IBUFDS_GTE3 (I,IB,ODIV2,O);
  parameter IOSTANDARD="DIFF_POD12_DCI";
  parameter SLEW="FAST";
  parameter IBUF_LOW_PWR="FALSE";
  input I,IB;
  output O,ODIV2;
  assign O = I;
  assign ODIV2 = I;
endmodule

module IBUF (O, I);
  output O;
  input I;
  assign O = I;
endmodule

module OBUFT (O, T, I);
  parameter IOSTANDARD="SSTL12_DCI";
  parameter SLEW="FAST";
  output O;
  input T,I;
  assign O = T? 1'bZ:I;
endmodule

module IOBUF (IO, O, T, I);
  parameter IOSTANDARD="POD12_DCI";
  parameter SLEW="FAST";
  parameter IBUF_LOW_PWR="FALSE";
  inout IO;
  output O;
  input T,I;
  assign IO = T? 1'bZ:I;
  assign O = IO;
endmodule

module IBUFDS (I, IB, O);
  parameter IOSTANDARD="DIFF_POD12_DCI";
  parameter SLEW="FAST";
  parameter IBUF_LOW_PWR="FALSE";
  input I,IB;
  output O;
  assign O = I;
endmodule

module IOBUFDS (IO, IOB, O, T, I);
  parameter IOSTANDARD="DIFF_POD12_DCI";
  parameter SLEW="FAST";
  parameter IBUF_LOW_PWR="FALSE";
  inout IO,IOB;
  output O;
  input T,I;
  assign IO = T? 1'bZ:I;
  assign IOB = ~IO;
  assign O = IO;
endmodule

module FD (Q, D, C);
  output reg Q;
  input C,D;
  always @(posedge C) Q <= D;
endmodule

module LUT1 (out,in);
  parameter DELAY=0;
  input in;
  output out;
  assign out = in;
endmodule

module LUT4 (out,in0,in1,in2,in3);
  parameter [15:0] LUTV=0;
  output out;
  input in0,in1,in2,in3;
  assign out = LUTV[{in3,in2,in1,in0}];
endmodule

module SRL16E (q, a0, a1, a2, a3, ena, clk, d);
  output q;
  input a0,a1,a2,a3;
  input ena,clk,d;

  reg [15:0] ram;
  always @(posedge clk) begin
    if (ena) ram <= { ram[14:0],d };
  end
  wire [3:0] a = {a3,a2,a1,a0};
  assign q = ram[a];
endmodule

module SRLC16E (q, c, a0, a1, a2, a3, ena, clk, d);
  output q,c;
  input a0,a1,a2,a3;
  input ena,clk,d;

  reg [15:0] ram;
  always @(posedge clk) begin
    if (ena) ram <= { ram[14:0],d };
  end
  wire [3:0] a = {a3,a2,a1,a0};
  assign q = ram[a];
  assign c = ram[15];
endmodule

module SRLC32E (Q, Q31, A, CE, CLK, D);
  output Q,Q31;
  input [4:0] A;
  input CE,CLK,D;

  reg [31:0] ram;
  always @(posedge CLK) begin
    if (CE) ram <= { ram[30:0],D };
  end
  assign Q = ram[A];
  assign Q31 = ram[31];
endmodule

module IDELAYE3 (CLK,CE,INC,RST,EN_VTC,IDATAIN,DATAOUT);
parameter DELAY_TYPE="VARIABLE", DELAY_VALUE=0, DELAY_FORMAT="COUNT";
input CLK,CE,INC,RST,EN_VTC,IDATAIN;
output DATAOUT;
assign DATAOUT = IDATAIN;
endmodule

module ISERDESE3 (D,CLKDIV,CLK,CLK_B,RST,Q,FIFO_RD_CLK,FIFO_EMPTY,FIFO_RD_EN);
parameter DATA_WIDTH=4, FIFO_ENABLE="TRUE", FIFO_SYNC_MODE="TRUE";
input CLKDIV,CLK,CLK_B,RST,FIFO_RD_CLK,FIFO_RD_EN;
output FIFO_EMPTY;
input D;
output [3:0] Q;
assign Q = D;
endmodule

module OSERDESE3 (OQ,CLKDIV,CLK,RST,T,T_OUT,D);
parameter DATA_WIDTH=4;
input CLKDIV,CLK,RST,T;
output T_OUT;
input [3:0] D;
output OQ;
assign OQ = D[0];
endmodule

module CLKPLLX (REFCLK,RST,CLKOUT);
parameter NOUT=1;
parameter FREQ_REF=100;
parameter FREQ_VCO=1200;
parameter FREQ_CLK0=100;
parameter FREQ_CLK1=100;
parameter FREQ_CLK2=100;
parameter FREQ_CLK3=100;
parameter FREQ_CLK4=100;
parameter FREQ_CLK5=100;
input REFCLK;
input RST;
output [NOUT-1:0] CLKOUT;
wire [7:0] clks;
`ifdef ALTERA
`else
`endif
assign CLKOUT = clks;
endmodule

module RAMBXX (CLKA,ENA,WEA_,SSRA,ADDRA,DIA_,DIPA_,DOA,DOPA,
               CLKB,ENB,WEB_,SSRB,ADDRB,DIB_,DIPB_,DOB,DOPB);

  parameter WIDTH_A=8;
  parameter WIDTH_B=8;
  parameter DEPTH=15; // ram bits = 2**DEPTH
  parameter FLG_A=0;
  parameter FLG_B=0;

  localparam NB=WIDTH_A/8, ND=(NB==0)?WIDTH_A:NB*8, NP=(NB==0)?1:NB;
  localparam MB=WIDTH_B/8, MD=(MB==0)?WIDTH_B:MB*8, MP=(MB==0)?1:MB;

  localparam LND=(ND>=64)? 6 : (ND>=32)? 5 : (ND>=16)? 4 : (ND>=8)? 3 : (ND==4)? 2 : (ND==2)? 1 : 0; // log2 port width
  localparam LMD=(MD>=64)? 6 : (MD>=32)? 5 : (MD>=16)? 4 : (MD>=8)? 3 : (MD==4)? 2 : (MD==2)? 1 : 0;

  localparam NA=DEPTH-LND;
  localparam MA=DEPTH-LMD;

  localparam NEL=(2**NA);
  localparam MEL=(2**MA);

  localparam NBE=(FLG_A&`BENA)? NP:1;
  localparam MBE=(FLG_B&`BENA)? MP:1;

  input CLKA,ENA,SSRA;
  input [NBE:0] WEA_;
  input [NA-1:0] ADDRA;
  input [ND-1:0] DIA_;
  input [NP-1:0] DIPA_;
  output [ND-1:0] DOA;
  output [NP-1:0] DOPA;

  input CLKB,ENB,SSRB;
  input [MBE-1:0] WEB_;
  input [MA-1:0] ADDRB;
  input [MD-1:0] DIB_;
  input [MP-1:0] DIPB_;
  output [MD-1:0] DOB;
  output [MP-1:0] DOPB;

  // handle write byte enables
  wire WEA,WEB;
  wire [ND-1:0] DIA;
  wire [NP-1:0] DIPA;
  wire [MD-1:0] DIB;
  wire [MP-1:0] DIPB;
  wire [ND-1:0] CDOA;
  wire [MD-1:0] CDOB;
  genvar i;
  generate
  if (FLG_A&`BENA) begin:aa
    assign WEA  = (WEA_!=0);
    for (i=0; i<NP; i=i+1) begin:c assign DIA[7+i*8:i*8]  = WEA_[i]? DIA_[7+i*8:i*8] : CDOA[7+i*8:i*8]; end
    assign DIPA = DIPA_;
  end
  else begin:ab
    assign WEA  = WEA_;
    assign DIA  = DIA_;
    assign DIPA = DIPA_;
  end
  if (FLG_B&`BENA) begin:ba
    assign WEB  = (WEB_!=0);
    for (i=0; i<MP; i=i+1) begin:c assign DIB[7+i*8:i*8]  = WEB_[i]? DIB_[7+i*8:i*8] : CDOB[7+i*8:i*8]; end
    assign DIPB = DIPB_;
  end
  else begin:bb
    assign WEB  = WEB_;
    assign DIB  = DIB_;
    assign DIPB = DIPB_;
  end
  endgenerate

  // infer from verilog

 /* verilator lint_off MULTIDRIVEN */
  reg [ND-1:0] ram [NEL-1:0];
  reg [NP-1:0] par [NEL-1:0];
 /* verilator lint_on MULTIDRIVEN */

  wire [ND-1:0] UDOA;
  wire [NP-1:0] UDOPA;
  wire [MD-1:0] UDOB;
  wire [MP-1:0] UDOPB;

  reg [ND-1:0] RDOA;
  reg [NP-1:0] RDOPA;
  reg [MD-1:0] RDOB;
  reg [MP-1:0] RDOPB;

  reg [NA-1:0] RADDRA;
  reg [MA-1:0] RADDRB;

  always @(posedge CLKA) begin
    if (ENA) begin
      if (WEA) ram[ADDRA] <= DIA;
      if (WEA) par[ADDRA] <= DIPA;
      RADDRA <= ADDRA;
    end
  end
  assign CDOA  = ram[ADDRA];
  assign UDOA  = ram[RADDRA];
  assign UDOPA = par[RADDRA];

generate

if (MD==ND) begin:c1to1
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[ADDRB] <= DIB;
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = ram[ ADDRB];
  assign UDOB  = ram[RADDRB];
end

if (MD==ND && NB!=0 && MB!=0) begin:c1to1p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[ADDRB] <= DIPB;
    end
  end
  assign UDOPB = par[RADDRB];
end

if (MD==2*ND) begin:c1to2
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[2*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[2*ADDRB+1] <= DIB[2*ND-1:1*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = {ram[2* ADDRB+1],ram[2* ADDRB+0]};
  assign UDOB  = {ram[2*RADDRB+1],ram[2*RADDRB+0]};
end

if (MD==2*ND && NB!=0 && MB!=0) begin:c1to2p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[2*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[2*ADDRB+1] <= DIPB[2*NP-1:1*NP];
    end
  end
  assign UDOPB = {par[2*RADDRB+1],par[2*RADDRB+0]};
end

if (MD==4*ND) begin:c1to4
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[4*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[4*ADDRB+1] <= DIB[2*ND-1:1*ND];
      if (WEB) ram[4*ADDRB+2] <= DIB[3*ND-1:2*ND];
      if (WEB) ram[4*ADDRB+3] <= DIB[4*ND-1:3*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = {ram[4* ADDRB+3],ram[4* ADDRB+2],ram[4* ADDRB+1],ram[4* ADDRB+0]};
  assign UDOB  = {ram[4*RADDRB+3],ram[4*RADDRB+2],ram[4*RADDRB+1],ram[4*RADDRB+0]};
end

if (MD==4*ND && NB!=0 && MB!=0) begin:c1to4p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[4*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[4*ADDRB+1] <= DIPB[2*NP-1:1*NP];
      if (WEB) par[4*ADDRB+2] <= DIPB[3*NP-1:2*NP];
      if (WEB) par[4*ADDRB+3] <= DIPB[4*NP-1:3*NP];
    end
  end
  assign UDOPB = {par[4*RADDRB+3],par[4*RADDRB+2],par[4*RADDRB+1],par[4*RADDRB+0]};
end

if (MD==8*ND) begin:c1to8
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[8*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[8*ADDRB+1] <= DIB[2*ND-1:1*ND];
      if (WEB) ram[8*ADDRB+2] <= DIB[3*ND-1:2*ND];
      if (WEB) ram[8*ADDRB+3] <= DIB[4*ND-1:3*ND];
      if (WEB) ram[8*ADDRB+4] <= DIB[5*ND-1:4*ND];
      if (WEB) ram[8*ADDRB+5] <= DIB[6*ND-1:5*ND];
      if (WEB) ram[8*ADDRB+6] <= DIB[7*ND-1:6*ND];
      if (WEB) ram[8*ADDRB+7] <= DIB[8*ND-1:7*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB = {ram[8* ADDRB+7],ram[8* ADDRB+6],ram[8* ADDRB+5],ram[8* ADDRB+4],
                 ram[8* ADDRB+3],ram[8* ADDRB+2],ram[8* ADDRB+1],ram[8* ADDRB+0]};
  assign UDOB = {ram[8*RADDRB+7],ram[8*RADDRB+6],ram[8*RADDRB+5],ram[8*RADDRB+4],
                 ram[8*RADDRB+3],ram[8*RADDRB+2],ram[8*RADDRB+1],ram[8*RADDRB+0]};
end
if (MD==8*ND && NB!=0 && MB!=0) begin:c1to8p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[8*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[8*ADDRB+1] <= DIPB[2*NP-1:1*NP];
      if (WEB) par[8*ADDRB+2] <= DIPB[3*NP-1:2*NP];
      if (WEB) par[8*ADDRB+3] <= DIPB[4*NP-1:3*NP];
      if (WEB) par[8*ADDRB+4] <= DIPB[5*NP-1:4*NP];
      if (WEB) par[8*ADDRB+5] <= DIPB[6*NP-1:5*NP];
      if (WEB) par[8*ADDRB+6] <= DIPB[7*NP-1:6*NP];
      if (WEB) par[8*ADDRB+7] <= DIPB[8*NP-1:7*NP];
    end
  end
  assign UDOPB = {par[8*RADDRB+7],par[8*RADDRB+6],par[8*RADDRB+5],par[8*RADDRB+4],
                  par[8*RADDRB+3],par[8*RADDRB+2],par[8*RADDRB+1],par[8*RADDRB+0]};
end

endgenerate

  always @(posedge CLKA) if (ENA) RDOA <= UDOA;
  always @(posedge CLKA) if (ENA) RDOPA <= UDOPA;
  assign DOA  = (FLG_A&`OREG)? RDOA : UDOA;
  assign DOPA = (FLG_A&`OREG)? RDOPA : UDOPA;

  always @(posedge CLKB) if (ENA) RDOB <= UDOB;
  always @(posedge CLKB) if (ENA) RDOPB <= UDOPB;
  assign DOB  = (FLG_B&`OREG)? RDOB : UDOB;
  assign DOPB = (FLG_B&`OREG)? RDOPB : UDOPB;

endmodule


module RAMB16_S36_S36 (CLKA,ENA,WEA,SSRA,ADDRA,DIA,DIPA,DOA,DOPA,
                       CLKB,ENB,WEB,SSRB,ADDRB,DIB,DIPB,DOB,DOPB);
  parameter NA=9,ND=32,NP=4;
  parameter MA=9,MD=32,MP=4;

  parameter [255:0] INIT_00=0, INIT_01=0, INIT_02=0, INIT_03=0, INIT_04=0, INIT_05=0, INIT_06=0, INIT_07=0, INIT_08=0, INIT_3F=0;

  localparam NEL=(2**NA);
  localparam MEL=(2**MA);

  input CLKA,ENA,WEA,SSRA;
  input [NA-1:0] ADDRA;
  input [ND-1:0] DIA;
  input [NP-1:0] DIPA;
  output [ND-1:0] DOA;
  output [NP-1:0] DOPA;

  input CLKB,ENB,WEB,SSRB;
  input [MA-1:0] ADDRB;
  input [MD-1:0] DIB;
  input [MP-1:0] DIPB;
  output [MD-1:0] DOB;
  output [MP-1:0] DOPB;

  reg [ND-1:0] ram [NEL-1:0];
  reg [NP-1:0] par [NEL-1:0];
  reg [NA-1:0] RADDRA;
  reg [MA-1:0] RADDRB;

`ifdef verilator
  reg inited;
  wire [255:0] tmp [9:0];
  assign tmp[0] = INIT_00;
  assign tmp[1] = INIT_01;
  assign tmp[2] = INIT_02;
  assign tmp[3] = INIT_03;
  assign tmp[4] = INIT_04;
  assign tmp[5] = INIT_05;
  assign tmp[6] = INIT_06;
  assign tmp[7] = INIT_07;
  assign tmp[8] = INIT_08;
  assign tmp[9] = INIT_3F;
  integer i,j;
  always @(posedge CLKA) begin
    inited <= 1;
    if (!inited) begin
      for (i=0; i<10; i=i+1) begin
      for (j=(i<9)? i : 63; j<100; j=j+100) begin
        ram[j*8+0]=tmp[i][31:00];
        ram[j*8+1]=tmp[i][63:32];
        ram[j*8+2]=tmp[i][95:64];
        ram[j*8+3]=tmp[i][127:96];
        ram[j*8+4]=tmp[i][159:128];
        ram[j*8+5]=tmp[i][191:160];
        ram[j*8+6]=tmp[i][223:192];
        ram[j*8+7]=tmp[i][255:224];
      end
      end
    end
  end
`endif

  always @(posedge CLKA) begin
    if (ENA) begin
      if (WEA) ram[ADDRA] <= DIA;
      if (WEA) par[ADDRA] <= DIPA;
      RADDRA <= ADDRA;
    end
//  end

//  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[ADDRB] <= DIB;
      if (WEB) par[ADDRB] <= DIPB;
      RADDRB <= ADDRB;
    end
  end

  assign DOA  = ram[RADDRA];
  assign DOPA = par[RADDRA];
  assign DOB  = ram[RADDRB];
  assign DOPB = par[RADDRB];

endmodule

module RAMB16_S9_S36 (CLKA,ENA,WEA,SSRA,ADDRA,DIA,DIPA,DOA,DOPA,
                      CLKB,ENB,WEB,SSRB,ADDRB,DIB,DIPB,DOB,DOPB);
  parameter NA=11,ND=8,NP=1;
  parameter MA=9,MD=32,MP=4;

  parameter [255:0] 
    INIT_00=0,INIT_01=0,INIT_02=0,INIT_03=0,INIT_04=0,INIT_05=0,INIT_06=0,INIT_07=0,INIT_08=0,INIT_09=0,INIT_0A=0,INIT_0B=0,INIT_0C=0,INIT_0D=0,INIT_0E=0,INIT_0F=0,
    INIT_10=0,INIT_11=0,INIT_12=0,INIT_13=0,INIT_14=0,INIT_15=0,INIT_16=0,INIT_17=0,INIT_18=0,INIT_19=0,INIT_1A=0,INIT_1B=0,INIT_1C=0,INIT_1D=0,INIT_1E=0,INIT_1F=0,
    INIT_20=0,INIT_21=0,INIT_22=0,INIT_23=0,INIT_24=0,INIT_25=0,INIT_26=0,INIT_27=0,INIT_28=0,INIT_29=0,INIT_2A=0,INIT_2B=0,INIT_2C=0,INIT_2D=0,INIT_2E=0,INIT_2F=0,
    INIT_30=0,INIT_31=0,INIT_32=0,INIT_33=0,INIT_34=0,INIT_35=0,INIT_36=0,INIT_37=0,INIT_38=0,INIT_39=0,INIT_3A=0,INIT_3B=0,INIT_3C=0,INIT_3D=0,INIT_3E=0,INIT_3F=0;

  localparam NEL=(2**NA);
  localparam MEL=(2**MA);

  input CLKA,ENA,WEA,SSRA;
  input [NA-1:0] ADDRA;
  input [ND-1:0] DIA;
  input [NP-1:0] DIPA;
  output [ND-1:0] DOA;
  output [NP-1:0] DOPA;

  input CLKB,ENB,WEB,SSRB;
  input [MA-1:0] ADDRB;
  input [MD-1:0] DIB;
  input [MP-1:0] DIPB;
  output [MD-1:0] DOB;
  output [MP-1:0] DOPB;

  reg [ND-1:0] ram [NEL-1:0];
  reg [NP-1:0] par [NEL-1:0];
  reg [NA-1:0] RADDRA;
  reg [MA-1:0] RADDRB;

  always @(posedge CLKA) begin
    if (ENA) begin
      if (WEA) ram[ADDRA] <= DIA;
      if (WEA) par[ADDRA] <= DIPA;
      RADDRA <= ADDRA;
    end
//  end

//  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[4*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[4*ADDRB+1] <= DIB[2*ND-1:1*ND];
      if (WEB) ram[4*ADDRB+2] <= DIB[3*ND-1:2*ND];
      if (WEB) ram[4*ADDRB+3] <= DIB[4*ND-1:3*ND];
      if (WEB) par[4*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[4*ADDRB+1] <= DIPB[2*NP-1:1*NP];
      if (WEB) par[4*ADDRB+2] <= DIPB[3*NP-1:2*NP];
      if (WEB) par[4*ADDRB+3] <= DIPB[4*NP-1:3*NP];
      RADDRB <= ADDRB;
    end
  end

  assign DOA  = ram[RADDRA];
  assign DOPA = par[RADDRA];
  assign DOB  = {ram[4*RADDRB+3],ram[4*RADDRB+2],ram[4*RADDRB+1],ram[4*RADDRB+0]};
  assign DOPB = {par[4*RADDRB+3],par[4*RADDRB+2],par[4*RADDRB+1],par[4*RADDRB+0]};

endmodule



module FIFO16 ( RST, EMPTY, FULL, RDCOUNT, WRCOUNT,
    WRCLK, WREN, DIP, DI, RDCLK, RDEN, DOP, DO);
  parameter DATA_WIDTH=32;
  parameter PARITY_WIDTH=4;
  parameter COUNT_WIDTH=9;
  parameter FIRST_WORD_FALL_THROUGH=0;
  parameter ALMOST_FULL_OFFSET=0;
  parameter ALMOST_EMPTY_OFFSET=0;
  localparam NEL=(2**COUNT_WIDTH);
  input RST,WRCLK,WREN,RDCLK,RDEN;
  output EMPTY,FULL;
  input [DATA_WIDTH-1:0] DI;
  output [DATA_WIDTH-1:0] DO;
  input [PARITY_WIDTH-1:0] DIP;
  output [PARITY_WIDTH-1:0] DOP;
  output [COUNT_WIDTH-1:0] WRCOUNT,RDCOUNT;
  assign EMPTY=1;
  assign FULL=0;

  wire L=0, H=1;

  reg [COUNT_WIDTH-1:0] WRCOUNT,RDCOUNT;

  always @(posedge WRCLK) begin
    if (RST) WRCOUNT <= 0;
    else if (WREN) WRCOUNT <= WRCOUNT+1;
  end
  always @(posedge RDCLK) begin
    if (RST) RDCOUNT <= 0;
    else if (RDEN) RDCOUNT <= RDCOUNT+1;
  end
  assign EMPTY = (RDCOUNT == WRCOUNT);
  assign FULL = (RDCOUNT == WRCOUNT+1);

  reg [31:0] ram [NEL-1:0];
  reg  [3:0] par [NEL-1:0];

  always @(posedge WRCLK) begin
    if (WREN) begin
      ram[WRCOUNT] <= DI;
      par[WRCOUNT] <= DIP;
    end
  end
  assign DO = ram[RDCOUNT];
  assign DOP = par[RDCOUNT];

endmodule


/* 

  Code Stubbs for compilation - not use

*/

module mgt_single ( gclk1,gclk2,pd,ts,tpmarst,rpmarst,tpcsrst,rpcsrst,tsync,rsync,
        tclk,rclk,okclk,tclko,rclko,tlock,rlock,tberr,rberr,
        txp,txn,rxp,rxn, txpol,rxpol, txdata,rxdata,
        dclk,den,daddr,dwr,dwrbus,drd,drdbus,
        cbe,cbi,cbo,combi,combo,tcrci,tcrco,rcrci,rcrco,testo);

  parameter WIDTH = 0;          // (0=8, 1=16, 2=32, 3=64)
  parameter PLLM  = 1;          // PLL multiplier
  parameter INDEX = -1;
  parameter CBMODE = "MASTER";  // master of slave_N_hop  N=1,2
  parameter BW  = 1<<WIDTH;      // Bus width in bytes
  parameter IBW = BW*8;
  parameter JBW = BW*9;

  localparam SYNC_DIVIDE = WIDTH;
  localparam CLOCK_DIVIDE = (BW==1)? 2 : (BW==2)? 1 : 3;

  input gclk1,gclk2,tclk,rclk,pd,ts,okclk;
  input tpmarst,rpmarst,tpcsrst,rpcsrst,tsync,rsync;
  output tclko,rclko;
  output tlock,rlock;
  output tberr,rberr;
  output txp,txn;
  input rxp,rxn;
  input txpol,rxpol;
  input [JBW-1:0] txdata;
  output [JBW-1:0] rxdata;
  input dclk,den,dwr,drd;
  input [7:0] daddr;
  input [15:0] dwrbus;
  output [15:0] drdbus;
  input cbe;
  input [4:0] cbi;
  output [4:0] cbo;
  input [15:0] combi;
  output [15:0] combo;
  input [70:0] tcrci,rcrci;
  output [31:0] tcrco,rcrco;
  output testo;

endmodule

module mgt_state (clk,rst,fast, lock,buferr, oclk,pmarst,pcsrst,sync,good);

parameter DELAY=1;
input clk,rst,fast,lock,buferr, oclk;
output reg pmarst,pcsrst,sync,good;

endmodule
