/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  DDR-4 Standard DRAM Controller (256Mby x 64b)

  MODE: 1=DDR/PIC5 2=DDR2/PIC6 3=DDR2/ZPPM 4=DDR2/VxM

  Pipelined command R/W queues each 16 bursts (64by) deep

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  5/15/2005

  This module uses 4 DRAMs to interface the 1333Mby/s input/output streams 
  to the 2666Mby/sec DDR interface.  All transfers must be naturally aligned.
  Burst transfers are 4, 16, 32, or 64 bytes in length.

  Use 64by banks and wait 11 cycles between same bank activations.

  Use delayed CAS to allow interleaved RAS for next bank

  Need 512 byte cache line controller to eliminate bank waits

  Memory control word:
  [31] nCS on 2nd cycle (config register)
  [30] FN on 1st cycle
  [29:28]=xfer 0-4by 1-16by 2-32by 3-64by
  [27:0]=ADDR [7:6]=BANK [2:0]=0

  [27:0]=ADDR [8:7]=BANK [2:0]=0

  PPC addr to SDRAM addr bits:
    64by banks:  crbl CBA9 8765 4321 0|987 6543 bb21 0xxx

  Losses:
    Refresh  20/1536
    TurnAround  5/64

  Cycle Timing:

  Busy
 !B
  A
  1 RAS      - new queue   bankwrite
  2 NOP      - new iqsel   bankwait   quad
  3 NOP  !B                            cquad
  4 CAS  A
  5 NOP  1 RAS
  6 CAS  2 NOP                        quad
         3 NOP                         cquad
         4 CAS
         5 NOP  1 RAS
         6 CAS  2 NOP
                3 NOP
                4 CAS
                5 NOP  1 RAS
                6 CAS  2 NOP
                       3 NOP
                       4 CAS
  1 RAS                5 NOP
  2 NOP                6 CAS


  acl fcmd0
1     fcmd1 arw            openrow
      fcmd2               RAS
      fcmd3 crw
  acl fcmd4 rd wr       access
      fcmd5 rd wr1        CAS
      fcmd6 rd wr2      access
      fcmd7 rd wr3        CAS   wack
8     fcmd8 rd wr4       
 tacl fcmd9 rd wr5        	wenj 
            rd wr6 		weni
            rd wr7 
                         
                         
                         
                          rack     
                          renl

                        bankok
                        !bankwait
18                      acl
                        fcmd1

  Memory Cache Line Mapping:

*/
`include "mdefs.h"
`include "ramdefs.h"

`define MOSTD #(.IOSTANDARD("SSTL12_DCI"),.SLEW("FAST"))
`define MDOSTD #(.IOSTANDARD("DIFF_SSTL12"),.SLEW("FAST"))
`define MIOSTD #(.IOSTANDARD("POD12_DCI"),.SLEW("FAST"),.IBUF_LOW_PWR("FALSE"))
`define MDIOSTD #(.IOSTANDARD("DIFF_POD12_DCI"),.SLEW("FAST"),.IBUF_LOW_PWR("FALSE"))

//`define READABLE 1

module sddrXram (
  clks,rst, ago,apd,acmd,afull,qrdym,qselm,
  ioclk, irdy,iena,ibus, ordy,oena,obus,
  mckp,mckn,mcke,mwe,mcas,mras,ma,mba,mdm,mdq,mdqsp,mdqsn,
  mcs,mrst,modt,mact, mclk,trs, test
  );

  parameter IBW=128;
  parameter MODE=8;
  parameter MBL=8;

  localparam NA=14;
  localparam NBA=3;
  localparam NDQS=MBL;		// DDR4 ram byte lanes 
  localparam NDQ=8*NDQS;	// DDR4 ram bit lanes

  localparam MBB=9;		// for 16K RAMs
  localparam IBB=10;		// for 16K RAMs
  localparam MBW=256;

  localparam RACKDLY=13;	// CL5=11 CL10=14 CL11+.5phase=15
  localparam WACKDLY=7;		// CWL9=7 CWL11=8

  input [39:0] clks;		// all memory clocks - see below

  // command group sync with ioclk
  input rst,ago,apd;		
  input [35:0] acmd;		// command address and modifiers
  output afull;			// cache output buffer almost full
  output [23:0] qrdym;
  input  [15:0] qselm;		// {RRi[7:1],qi_sel[2:0],ro_sel[2:0],ri_sel[2:0]}

  // crossbar bus sync with ioclk
  input  ioclk, iena,oena;
  input  irdy,ordy;
  input  [IBW-1:0] obus;
  output [IBW-1:0] ibus;

  // actual DRAM signals
  output mckp,mckn,mcke,mwe,mcas,mras;
  output [NA-1:0] ma; 
  output [NBA-1:0] mba;
  output [NDQS-1:0] mdm;
  inout  [NDQ-1:0] mdq;
  inout  [NDQS-1:0] mdqsp,mdqsn;
  output mcs,mrst,modt,mact;

  // possible feedback signals - not used
  input mclk;			
  output [2:0] trs;
  output [7:0] test;

  wire H=1, L=0;

  // debug signals
  reg [3:0] ttt; always @(posedge aclk) ttt <= clks[35:32];
  wire trsr  = ttt[0];	// MEMTSTT
  wire trsw  = ttt[1];	// MEMTSTW
  wire tswp  = L; // ttt[0];	// MEMTSTT
  wire txdt  = L; // ttt[0];	// MEMTSTT
  wire twrt  = L; // ttt[1];	// MEMTSTW
  wire tpad  = L; //ttt[2];	// MEMTSTP
  wire tblk  = ttt[2];	// MEMTSTP
  wire tlbt  = ttt[3];	// MEMTSTX

  // per chip masks
  reg [3:0] pcm; always @(posedge aclk) pcm <= ~clks[39:36];
 
  // clocks
  wire aclk = clks[0];
  wire rclk = clks[1];
  wire wclk = clks[2];
  wire sclk  = clks[3];
  wire aclk2 = clks[4];
  wire rclk2 = clks[5];
  wire wclk2 = clks[6];
  wire adjcyc = clks[7];
  wire [4:0] dadj = {!adjcyc,clks[11:8]};
  wire [4:0] cadj = {!adjcyc,clks[15:12]};
  wire [15:0] rcfg = clks[31:16];
  reg arst;    always @(posedge aclk) arst    <= rst;		// note rclk2 & wclk2 down for 24 clks at this point
  wire rstn;  delaypipe #(60) startup (aclk,H,rstn);
  wire arstd; delaypipe #(12,1) clkdp (aclk,arst,arstd);
  wire arste; delaypipe #(60,1) clkep (aclk,arst,arste);
  wire arstf; delaypipe #(40,1) clkfp (aclk,arst,arstf);
  reg clkrst;  always @(posedge aclk) clkrst  <= arst && !arstd;
  reg clkrstd; always @(posedge aclk) clkrstd <= arst && !arstf;
  reg aclkrst; always @(posedge aclk) aclkrst <= clkrst;
  reg rclkrst; always @(posedge rclk) rclkrst <= trsr? clkrst : clkrstd;
  reg wclkrst; always @(posedge wclk) wclkrst <= trsw? clkrst : clkrstd;
  wire [2:0] aclks = {aclkrst,aclk2,aclk};
  wire [2:0] rclks = {rclkrst,rclk2,rclk};
  wire [2:0] wclks = {wclkrst,wclk2,wclk};

  genvar i;

  wire rack,wack,wen,weni,wenj;
  wire renl,renu,wenl,wenu,ienax,oenax;
  wire [MBW-1:0] rdata,wdata;
  wire [IBB-1:0] iadrx,oadrx;
  wire [MBB-1:0] rcntl,rcntu,wcntl,wcntu;

  wire acl,refresh;
  wire [35:0] qcmd;
  wire [31:0] tdata;
  wire [7:0] stest;
  scheduler #(2,IBB,MBB,RACKDLY,WACKDLY) cnt (
        ioclk,arst, ago,acmd,afull, qrdym,qselm,
        ioclk,irdy,iena,ienax,iadrx,
	ioclk,ordy,oena,oenax,oadrx,
        aclk,acl,qcmd,refresh, 
	rclk,rack,renl,rcntl,renu,rcntu, 
	wclk,wack,wenl,wcntl,wenu,wcntu,
        sclk,rcfg,stest); //,tdata);

  // DMAR cache data array 
  sdpram #(16,MBW,IBW,0,0) rl (rclk,H, rcntl,renl,rdata,  ioclk,H, iadrx,ibus);

  // DMAW cache data array 
  sdpram #(16,IBW,MBW,0,`OREG) wl (ioclk,H, oadrx,oenax,obus,  wclk,H, wcntl,wdata);

  // test vectors
  reg wrst; always @(posedge wclk) wrst <= acl;
  reg  [2:0] wtst; always @(posedge wclk) if (wrst) wtst <= 1; else wtst <= wtst+1;
  wire [3:0] wtst0 = {wtst,L}, wtst1 = {wtst,H};
  wire [127:0] wpat = { {8{~wtst1}},{8{wtst1}}, {8{~wtst0}},{8{wtst0}} };
  reg  [2:0] rtst; always @(posedge rclk) rtst <= wtst;
  wire [3:0] rtst0 = {rtst,L}, rtst1 = {rtst,H};
  wire [127:0] rpat = { {8{~rtst1}},{8{rtst1}}, {8{~rtst0}},{8{rtst0}} };
  wire [MBW-1:0] rblk;

  // register the output data for routing purposes
  reg [MBW-1:0] vdata;
  always @(posedge wclk) vdata <= tlbt? {wpat,wpat} : wdata;
  wire [MBW-1:0] xdata = vdata;

  // register the input data for routing purposes
  reg [MBW-1:0] sdata;
  wire [MBW-1:0] ydata;
  always @(posedge rclk) sdata <= tblk? rblk : ydata;
  assign rdata = sdata;

  reg rd,rd1,rd2,rd3,rd4;
  reg wr,wr1,wr2,wr3,wr4,wr5;

  reg [NA-1:0] fma;
  reg [NBA-1:0] fmba;
  reg fmwe,fmrst,fmcs,fmodt,fmdm,fmact,fmcke,fmras,fmcas;
  reg fcmd1,fcmd2,fcmd3,fcmd4,fcmd5,fcmd6,fcmd7,fcmd8,fcmdx;
  reg arw,crw,acfg,dual,cdual,zqcl;
  reg [NBA-1:0] bank,cbank;
  reg [31:2] addr,caddr,daddr;

  // burst DMA state machine
  always @(posedge aclk) begin
    if (acl) arw   <= qcmd[35];		// read/write
    if (acl) acfg  <= qcmd[34];		// configuration command
    if (acl) dual  <= qcmd[33:32]==2;	// 128by length
    if (acl) addr  <= qcmd[31:2];	// start RAM address
    if (acl) bank  <= qcmd[34]? qcmd[8:6] : {qcmd[7],qcmd[9:8]};	// want to alter bank groups every other slot for RRD_S vs RRD_L
    if (acl) zqcl  <= qcmd[34] && (qcmd[8:6]==7);
    if (fcmd2) crw  <= arw;
    if (fcmd2) cbank <= bank;
    if (fcmd2) caddr <= addr;
    if (fcmd2) cdual <= dual;
    if (!rstn) fmrst <= L; else if (arst^arstd) fmrst <= !arst;
    if (arst) fmodt <= H; else fmodt <= H;				// active H
    if (arst) fmdm  <= H; else fmdm  <= H;				// active H
    fmcke <= !arste;							// active H
    if (fcmd4) daddr <= caddr;
  end

  // command state machine
  wire access = (fcmd4 || (cdual&&fcmd6)) && !acfg;
  wire autopre = cdual? fcmd6 : fcmd4;
  wire openrow  = fcmd1;
  wire precharge = addr[13];
  wire fcmd0 = acl && !fcmd1;
  wire bchop = H;	// no burst chop
  reg openrow_,access_,access__,autopre_;
  always @(posedge aclk) begin
    fcmd1 <= fcmd0;
    fcmd2 <= fcmd1;
    fcmd3 <= fcmd2;
    fcmd4 <= fcmd3; 
    fcmd5 <= fcmd4; 
    fcmd6 <= fcmd5; 
    fcmd7 <= fcmd6; 
    fcmd8 <= fcmd7; 
    fcmdx <= fcmd0|fcmd1|fcmd2|fcmd3|fcmd4|fcmd5|fcmd6|fcmd7;
`ifdef READABLE
    if (openrow) begin			// activate row
      fmact <= acfg;			// {ras,cas,we} are address 16:14 when act 
      fmras <= acfg? zqcl : L;
      fmcas <= acfg? precharge|zqcl : addr[31];
      fmwe  <= acfg? L : addr[30];
      fmba  <= bank;
      fma   <= addr[29:16];
    end else if (access) begin		// column read|write
      fmact <= H;
      fmras <= H;
      fmcas <= L;
      fmwe  <= crw;
      fmba  <= cbank;
      fma   <= {L,bchop,L,autopre, caddr[15:10],(caddr[6]^fcmd6),3'b000};	// on 64by cache line boundaries only
    end else if (refresh) begin		// autorefresh
      fmact <= H;
      fmras <= L;
      fmcas <= L;
      fmwe  <= H;
    end else begin    			// nop
      fmact <= H;
      fmras <= H;
      fmcas <= H;
      fmwe  <= H;
    end
`else
    // help the state machine encoder
    openrow_ <= fcmd0;
    access__ <= (fcmd1&&tpad&&arw) && !acfg;
    access_  <= (fcmd3 || (cdual&&fcmd5)) && !acfg && !tlbt;
    autopre_ <= cdual? fcmd5 : fcmd3;
    fmras    <= openrow_? (acfg? zqcl:L) : access_? H : refresh? L : H;
    fmcas    <= openrow_? (acfg? precharge|zqcl:addr[31]) : access_? L : refresh? L : H;
    fmwe     <= openrow_? (acfg? L:addr[30]) : access_? crw : refresh? H : H;
    fmba     <= openrow_? bank : cbank;
    fma      <= openrow_? addr[29:16] : {L,bchop,L,autopre_, caddr[15:10],(caddr[6]^fcmd6),3'b000};
    fmcs     <= !(openrow_ || access_ || access__ || refresh);
    fmact    <= openrow_? acfg : H;
`endif
    rd  <= (fcmd3||fcmd5) && crw;			// always read 64 by
    rd1 <= rd; rd2 <= rd1; rd3 <= rd2; rd4 <= rd3;	// rd1 aligns with fmcas
    wr  <= (fcmd3||(cdual&fcmd5)) && !crw;
    wr1 <= wr; wr2 <= wr1; wr3 <= wr2; wr4 <= wr3; wr5 <= wr4;	// wd1 aligns with fmcas
  end

  delaypipe #(RACKDLY-4,1) rackdp (aclk,rd1|rd2,rack);
  delaypipe #(WACKDLY-5,1) wackdp (aclk,wr1|wr2,wack);	// CWL9
  delaypipe #(WACKDLY-2,1) wendp  (aclk,(!acfg)&(   wr1|wr2),wen);
  delaypipe #(WACKDLY-2,1) wenidp (aclk,(!acfg)&(   wr1|wr2),weni);
  delaypipe #(WACKDLY-2,1) wenjdp (aclk,(!acfg)&(wr|wr1|wr2),wenj);

  // output enables - one early last half, one late first half
  wire oenk; delaypipe #(WACKDLY-3,1) oenkdp (aclk,   wr1|wr2|wr3|wr4|wr5,oenk); 
  wire wenk; delaypipe #(WACKDLY-3,1) wenkdp (aclk,   wr|wr1|wr2|wr3|wr4|wr5,wenk); 
  reg woenk; always @(posedge wclk) woenk <= wenk || tlbt;

  // timing recovery signals
  wire [NDQS-1:0] rdclk,mrclk;

  // block ram test emulator
`ifdef DDR_BLK_EMU
  wire        bwena; delaypipe #(WACKDLY-2,1,1 ) bwena_ (wclk,wr1|wr2,bwena); 
  wire [13:0] bwadr; delaypipe #(WACKDLY-2,1,14) bwadr_ (wclk,{daddr[23:21],daddr[15:7],daddr[6]^(fcmd7|fcmd8),wr2},bwadr); 
  wire [13:0] bradr; delaypipe #(RACKDLY-6,1,14) bradr_ (rclk,{daddr[23:21],daddr[15:7],daddr[6]^(fcmd7|fcmd8),rd2},bradr); 
  sdpram #(512,MBW,MBW,0,`OREG) bram (wclk,H, bwadr,bwena,xdata,  rclk,H, bradr,rblk);
`else
  assign rblk = ydata;
`endif

  // chip reset
  wire mrst_ = fmrst && rstn;
  obuftN bmrst (mrst,L,mrst_);

  // DDR clocks (180 phase)
  wire mclkx;
  qdr_od bmck (mckp,mckn, aclks, H,L,H,L, mclkx);

  // control signals
  qdr_o bmcke (mcke, aclks, fmcke,fmcke,fmcke,fmcke);
  qdr_o bmras (mras, aclks, fmras,fmras,fmras,fmras);
  qdr_o bmcas (mcas, aclks, fmcas,fmcas,fmcas,fmcas);
  qdr_o bmact (mact, aclks, fmact,fmact,fmact,fmact);
  qdr_o bmwe  (mwe,  aclks, fmwe,fmwe,fmwe,fmwe);

  // chip select
  qdr_o bmcs (mcs, aclks, H,fmcs,fmcs,H);

  // on die termination select 
  qdr_o bmodt (modt, aclks, fmodt,fmodt,fmodt,fmodt);

  generate

  // address signals
  for (i=0; i<NA; i=i+1) begin:bma
    qdr_o inst  (ma[i], aclks, fma[i],fma[i],fma[i],fma[i]);
  end

  // bank signals
  for (i=0; i<NBA; i=i+1) begin:bmba
    qdr_o inst (mba[i], aclks, fmba[i],fmba[i],fmba[i],fmba[i]);
  end

  // data strobes
  for (i=0; i<NDQS; i=i+1) begin:db
    wire [3:0] rdqs;
    assign mrclk[i] = rclk2;

    qdr_o odm (mdm[i], aclks, fmdm,fmdm,fmdm,fmdm);
    
    localparam k = i>>1;
    qdr_ioc ioc  (mdqsp[i],mdqsn[i],cadj,pcm[k], rdclk[i], aclks,rdqs[0],rdqs[1],rdqs[2],rdqs[3],
                           oenk,oenk,oenk,oenk,          aclks,H,~weni,H,~wenj);

    // 32b chip1&2 ReFeReFe , 32b chip3&4 ReFeReFe, 
    localparam j = (i>=4)? i*8+96 : i*8;
    qdr_iod iod (mdq[i*8+7:i*8],dadj,pcm[k], mrclk[i], rclks,ydata[j+7:j+0],ydata[j+39:j+32],ydata[j+71:j+64],ydata[j+103:j+96],
                            woenk,woenk,woenk,woenk, wclks,xdata[j+7:j+0],xdata[j+39:j+32],xdata[j+71:j+64],xdata[j+103:j+96]);  
  end

  endgenerate

  assign trs  = {L,L,L}; 

  reg ccc; always @(posedge rclk) ccc <= !ccc;

//assign test = {oenk,refresh,access_,openrow_,cdual,wenl,oenax,wack};
//assign test = {oenk,refresh,access_,openrow_,dadj[4:1]};
//assign test = {oeni,refresh,acl,ago,jjj};
//assign test = {renl,wenl,acfg,wen, dual,ccc,jjj[1:0]};
//assign test = stest;
//assign test = {dual,woenk,refresh,fmrst,rack,wack,fcmdx,ccc};
assign test = {dual,woenk,refresh,fmrst,rack,wack,fcmdx,ccc};


endmodule

// quad data rate clocked output buffer
module qdr_o (out, clks, in0,in1,in2,in3);
  output out;
  input [2:0] clks;
  input in0,in1,in2,in3;

  wire H=1, L=0;
  wire clk=clks[0];
  wire clk2=clks[1];
  wire clkrst=clks[2];
  wire ff,tt;

  OBUFT `MOSTD b0 (.I(ff), .T(tt), .O(out) );

  OSERDESE3 #(.DATA_WIDTH(4))
    f0 (.OQ(ff),.T_OUT(tt), .CLKDIV(clk),.CLK(clk2),.RST(clkrst), .T(L), .D({in3,in2,in1,in0}));

endmodule

// quad data rate clocked output buffer with DS
module qdr_od (outp,outn, clks, in0,in1,in2,in3, out);
  output outp,outn;
  input [2:0] clks;
  input in0,in1,in2,in3;
  output out;

  wire H=1, L=0;

  wire clk=clks[0];
  wire clk2=clks[1];
  wire clkrst=clks[2];
  wire ff,tt;

  IOBUFDS `MDOSTD b0 (.I(ff),.T(tt),.IO(outp),.IOB(outn),.O(out) );

  OSERDESE3 #(.DATA_WIDTH(4))
    f0 (.OQ(ff),.T_OUT(tt), .CLKDIV(clk),.CLK(clk2),.RST(clkrst), .T(L), .D({in3,in2,in1,in0}));

endmodule

// double data rate clocked input/output buffer
module qdr_iod (io,cadj,pcm, rstrobe, rclks,rdata0,rdata1,rdata2,rdata3, ena0,ena1,ena2,ena3, wclks,wdata0,wdata1,wdata2,wdata3);
  inout [7:0] io;
  input [4:0] cadj;
  input pcm,rstrobe;
  input [2:0] rclks,wclks;
  input ena0,ena1,ena2,ena3;
  output [7:0] rdata0,rdata1,rdata2,rdata3;
  input  [7:0] wdata0,wdata1,wdata2,wdata3;

  wire H=1, L=0;

  wire rclk=rclks[0];
  wire rclk2=rclks[1];
  wire rclkrst=rclks[2];
  wire wclk=wclks[0];
  wire wclk2=wclks[1];
  wire wclkrst=wclks[2];

  wire sclk = cadj[0];
  wire srst = cadj[1];
  wire sadj = cadj[2] && pcm;
  wire supd = cadj[3];
  wire svtc = cadj[4];

  wire [7:0] fempty;
  reg rfen; always @(posedge rclk) rfen <= !rclkrst;
//  wire rfen = !fempty[0];

genvar i;
generate
for (i=0; i<8; i=i+1) begin:dq
  wire rff,rffd,wff,tff;
  wire [3:0] sdata;

  IOBUF `MIOSTD odq (.I(wff),.T(tff),.IO(io[i]),.O(rff) );

  IDELAYE3 dly (.CLK(sclk),.CE(sadj),.INC(supd),.RST(srst),.EN_VTC(svtc), .IDATAIN(rff),.DATAOUT(rffd));
  defparam dly.DELAY_TYPE = "VARIABLE";
  defparam dly.DELAY_VALUE = 0;
  defparam dly.DELAY_FORMAT = "COUNT";

  ISERDESE3 #(.DATA_WIDTH(4),.FIFO_ENABLE("TRUE"),.FIFO_SYNC_MODE("TRUE"))
    brff (.D(rffd), .CLKDIV(rclk),.CLK(rclk2),.CLK_B(~rclk2),.RST(rclkrst), .Q(sdata), .FIFO_RD_CLK(rclk),.FIFO_EMPTY(fempty[i]),.FIFO_RD_EN(rfen));

`ifdef SD4_CL10
  assign {rdata3[i],rdata2[i],rdata1[i],rdata0[i]} = sdata;
`else
  reg  [3:0] qdata; always @(posedge rclk) qdata <= sdata;
//  assign {rdata3[i],rdata2[i],rdata1[i],rdata0[i]} = {sdata[3],sdata[2],sdata[1],sdata[0]};
//  assign {rdata3[i],rdata2[i],rdata1[i],rdata0[i]} = {sdata[0],qdata[3],qdata[2],qdata[1]};
//  assign {rdata3[i],rdata2[i],rdata1[i],rdata0[i]} = {sdata[1],sdata[0],qdata[3],qdata[2]};
  assign {rdata3[i],rdata2[i],rdata1[i],rdata0[i]} = {sdata[2],sdata[1],sdata[0],qdata[3]};
`endif

  OSERDESE3 #(.DATA_WIDTH(4))
    bwff (.OQ(wff), .CLKDIV(wclk),.CLK(wclk2),.RST(wclkrst), .T(~ena0),.T_OUT(tff), .D({wdata3[i],wdata2[i],wdata1[i],wdata0[i]}));
end
endgenerate

endmodule

// double data rate clocked input/output buffer w/ clock input
module qdr_ioc (iop,ion,cadj,pcm, rdata, rclks,rdata0,rdata1,rdata2,rdata3, ena0,ena1,ena2,ena3, wclks,wdata0,wdata1,wdata2,wdata3);
  inout iop,ion;
  input [4:0] cadj;
  input pcm;
  output rdata;
  input [2:0] rclks,wclks;
  input ena0,ena1,ena2,ena3;
  output rdata0,rdata1,rdata2,rdata3;
  input wdata0,wdata1,wdata2,wdata3;

  wire H=1, L=0;

  wire rff,rffd,wff,wffd,tff;
  wire rclk=rclks[0];
  wire rclk2=rclks[1];
  wire rclkrst=rclks[2];
  wire wclk=wclks[0];
  wire wclk2=wclks[1];
  wire wclkrst=wclks[2];

  wire sclk = cadj[0];
  wire srst = cadj[1];
  wire sadj = cadj[2] && pcm;
  wire supd = cadj[3];
  wire svtc = cadj[4];

  IOBUFDS `MDIOSTD odq (.I(wff),.T(tff),.IO(iop),.IOB(ion),.O(rff) );

  IDELAYE3 dlyi (.CLK(sclk),.CE(sadj),.INC(supd),.RST(srst),.EN_VTC(svtc), .IDATAIN(rff),.DATAOUT(rffd));
  defparam dlyi.DELAY_TYPE = "VARIABLE";
  defparam dlyi.DELAY_VALUE = 0;
  defparam dlyi.DELAY_FORMAT = "COUNT";

  ISERDESE3 #(.DATA_WIDTH(4))
    brff (.D(rffd), .CLKDIV(rclk),.CLK(rclk2),.CLK_B(~rclk2),.RST(rclkrst), .Q({rdata3,rdata2,rdata1,rdata0}));
  assign rdata = rffd;

  OSERDESE3 #(.DATA_WIDTH(4))
    bwff (.OQ(wff), .CLKDIV(wclk),.CLK(wclk2),.RST(wclkrst), .T(~ena0),.T_OUT(tff), .D({wdata3,wdata2,wdata1,wdata0}));

endmodule

