/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  Various Multiplexors

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

*/

`include "mdefs.h"

module muxes();

endmodule

// Parameterized Muxes

module muxMxN (out, in, sel);
  parameter M=1;
  parameter N=1;
  output [N-1:0] out;
  input [M*N-1:0] in;
  input [5:0] sel;
  muxMxNpec #(M,N,0) m (out, in, sel, 1'b0, 1'b1, 1'b0);
endmodule

module muxMxNp (out, in, sel, clk);
  parameter M=1;
  parameter N=1;
  parameter P=1;
  output [N-1:0] out;
  input [M*N-1:0] in;
  input [5:0] sel;
  input clk;
  muxMxNpec #(M,N,P) m (out, in, sel, clk, 1'b1, 1'b0);
endmodule

module muxMxNpe (out, in, sel, clk, ena);
  parameter M=1;
  parameter N=1;
  parameter P=1;
  output [N-1:0] out;
  input [M*N-1:0] in;
  input [5:0] sel;
  input clk,ena;
  muxMxNpec #(M,N,P) m (out, in, sel, clk, ena, 1'b0);
endmodule

module muxMxNpec (out, in, sel, clk, ena, clr);
  parameter M=2;
  parameter N=1;
  parameter P=1;

  localparam L=(M>32)? 6 : (M>16)? 5 : (M>8)? 4 : (M>4)? 3 : (M>2)? 2 : (M>1)? 1 : 0;
  localparam M1=M-1;
  localparam M2=(1<<L);

  output [N-1:0] out;
  input [M*N-1:0] in;
  input [L-1:0] sel;
  input clk,ena,clr;

  wire [N-1:0] inw [M2-1:0];
  genvar i;
  generate
  for (i=0; i<M; i=i+1) begin:blk
    assign inw[i] = in[i*N+N-1:i*N];
  end
  for (i=M; i<M2; i=i+1) begin:blkx
    assign inw[i] = in[M1*N+N-1:M1*N];
  end
  endgenerate

  wire [N-1:0] outp;
  assign outp = inw[sel];
  reg [N-1:0] outr; always @(posedge clk) if (clr) outr <= 0; else if (ena) outr <= outp;
  assign out = (P==0)? outp : outr;

endmodule

module muxTMxN (out, in, sel);
  parameter M=1;
  parameter N=1;
  output [N-1:0] out;
  input [M*N-1:0] in;
  input [M-1:0] sel;

  wire [N-1:0] tmp;

  genvar i,j;
  generate
  for (i=0; i<N; i=i+1) begin:blk1
   for (j=0; j<M; j=j+1) begin:blk2
    assign tmp[i] = sel[j] & in[N*j+i];
   end
  end
  endgenerate

  assign out = tmp;

endmodule

module muxTxN (out, in, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in;
  input sel;

`ifdef verilator
  assign out = sel? in : {N{1'bZ}};
`else
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    BUFE inst (out[i], sel, in[i]);
  end
  endgenerate
`endif

endmodule

`ifdef VIVADO
module BUFE (out, sel, in);
  output out;
  input sel,in;
  assign out = sel? in : 1'bZ;
endmodule
`endif

// Standard Muxes

module mux2xN (out, in0,in1, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1;
  input sel;
  assign out = sel? in1 : in0;
endmodule

module mux1xNp (out, in0, clk);
  parameter N=1;
  output reg [N-1:0] out;
  input [N-1:0] in0;
  input clk;
  always @ (posedge clk) out <= in0;
endmodule

module mux2xNp (out, in0,in1, sel,clk);
  parameter N=1;
  output reg [N-1:0] out;
  input [N-1:0] in0,in1;
  input sel;
  input clk;
  always @ (posedge clk) out <= sel? in1 : in0;
endmodule

module mux2xNpe (out, in0,in1, sel,clk,ena);
  parameter N=1;
  output reg [N-1:0] out;
  input [N-1:0] in0,in1;
  input sel;
  input clk,ena;
  always @ (posedge clk) if (ena) out <= sel? in1 : in0;
endmodule

module mux3xN (out, in0,in1,in2, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2;
  input [1:0] sel;
  muxMxNpec #(3,N,0) m (out, {in2,in1,in0}, sel,1'b0,1'b1,1'b0);
endmodule

module mux3xNp (out, in0,in1,in2, sel,clk);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2;
  input [1:0] sel;
  input clk;
  muxMxNpec #(3,N,1) m (out, {in2,in1,in0}, sel,clk,1'b1,1'b0);
endmodule

module mux4xN (out, in0,in1,in2,in3, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3;
  input [1:0] sel;
  muxMxNpec #(4,N,0) m (out, {in3,in2,in1,in0}, sel,1'b0,1'b1,1'b0);
endmodule

module mux4xNp (out, in0,in1,in2,in3, sel,clk);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3;
  input [1:0] sel;
  input clk;
  muxMxNpec #(4,N,1) m (out, {in3,in2,in1,in0}, sel,clk,1'b1,1'b0);
endmodule

module mux4xNpe (out, in0,in1,in2,in3, sel,clk,ena);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3;
  input [1:0] sel;
  input clk,ena;
  muxMxNpec #(4,N,1) m (out, {in3,in2,in1,in0}, sel,clk,ena,1'b0);
endmodule

module mux4xNpec (out, in0,in1,in2,in3, sel,clk,ena,clr);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3;
  input [1:0] sel;
  input clk,ena,clr;
  muxMxNpec #(4,N,1) m (out, {in3,in2,in1,in0}, sel,clk,ena,clr);
endmodule

module mux5xN (out, in0,in1,in2,in3,in4, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3,in4;
  input [2:0] sel;
  muxMxNpec #(5,N,0) m (out, {in4,in3,in2,in1,in0}, sel,1'b0,1'b1,1'b0);
endmodule

module mux5xNp (out, in0,in1,in2,in3,in4, sel,clk);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3,in4;
  input [2:0] sel;
  input clk;
  muxMxNpec #(5,N,1) m (out, {in4,in3,in2,in1,in0}, sel,clk,1'b1,1'b0);
endmodule

module mux8xN (out, in0,in1,in2,in3,in4,in5,in6,in7, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3,in4,in5,in6,in7;
  input [2:0] sel;
  muxMxNpec #(8,N,0) m (out, {in7,in6,in5,in4,in3,in2,in1,in0}, sel,1'b0,1'b1,1'b0);
endmodule

module mux16xN (out, in0,in1,in2,in3,in4,in5,in6,in7,in8,in9,inA,inB,inC,inD,inE,inF, sel);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3,in4,in5,in6,in7,in8,in9,inA,inB,inC,inD,inE,inF;
  input [3:0] sel;
  muxMxNpec #(16,N,0) m (out, {inF,inE,inD,inC,inB,inA,in9,in8,in7,in6,in5,in4,in3,in2,in1,in0}, sel,1'b0,1'b1,1'b0);
endmodule

module mux8xNp (out, in0,in1,in2,in3,in4,in5,in6,in7, sel,clk);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1,in2,in3,in4,in5,in6,in7;
  input [2:0] sel;
  input clk;
  muxMxNpec #(8,N,1) m (out, {in7,in6,in5,in4,in3,in2,in1,in0}, sel,clk,1'b1,1'b0);
endmodule

// Specialty Muxes

// mixes 2 complex crossbar streams (fmt? CX=3 CL=2 CI=1 CB=0);
module mix2xNp (out, in0,in1, sel,clk, mix,fmt);
  parameter N=1;
  output reg [N-1:0] out;
  input [N-1:0] in0,in1;
  input sel;
  input clk;
  input mix;		// A/B mix
  input [1:0] fmt;	// 8b=0 16b=1 32b=2 64b=3
  genvar i;
  generate
  for (i=0; i<N; i=i+8) begin:blk
    wire select = !mix? sel : (fmt==3)? (i&64)!=0 : (fmt==2)? (i&32)!=0 : (fmt==1)? (i&16)!=0 : (i&8)!=0;
    always @(posedge clk) out[i+7:i] <= select? in1[i+7:i] : in0[i+7:i];
  end
  endgenerate
endmodule

module cmix2xNp (out, in0,in1, sel,clk, mix,pha);
  parameter N=1;
  output [N-1:0] out;
  input [N-1:0] in0,in1;
  input clk,sel,mix,pha;
  wire [2*N-1:0] cx;
  genvar i;
  generate
  for (i=0; i<N; i=i+16) begin:blk
    assign cx[15+i*2:   i*2] = in0[15+i:i];
    assign cx[31+i*2:16+i*2] = in1[15+i:i];
  end
  endgenerate
  mux4xNp #(N) cm (out, in0,in1,cx[N-1:0],cx[2*N-1:N], {mix,mix?pha:sel},clk);
endmodule

// mixes 2 complex crossbar streams fmt = CI|SL interleaved
`ifdef verilator
module mix2xNci (out, in0,in1, sel, mix);
  parameter N=1;
  parameter NC=N/32;
  output [N-1:0] out;
  input [N-1:0] in0,in1;
  input sel;
  input mix;		// A/B mix
  wire [2*N-1:0] in = {in1,in0};
  genvar i;
  generate
  for (i=0; i<NC; i=i+1) begin:blk
    localparam j = i%2;
    localparam k = i/2;
    localparam ia = i;
    localparam ib = i+NC;
    localparam ic = k+j*NC;
    localparam id = k+j*NC+NC/2;
    wire [31:0] a = in[31+ia*32:ia*32];
    wire [31:0] b = in[31+ib*32:ib*32];
    wire [31:0] c = in[31+ic*32:ic*32];
    wire [31:0] d = in[31+id*32:id*32];
    wire [31:0] x = mix? {sel?d:c} : {sel?b:a};
    assign out[31+i*32:i*32] = x;
  end
  endgenerate
endmodule
`endif


// for DTDM tuner control readback
module mux6x8pe (out, in0,in1,in2,in3, in4,in5, page,pagem,pagep,clk,ce);
  output reg [7:0] out;
  input [7:0] in0,in1,in2,in3,in4,in5;
  input [7:0] page;
  input clk,ce,pagem,pagep;
  wire [2:0] sel;

  assign sel[2] = (pagem || pagep);
  assign sel[1] = (page[2] || page[6] || page[3] || page[7]);  
  assign sel[0] = (page[1] || page[5] || page[3] || page[7]) || pagep;

  always @ (posedge clk) begin
    if (ce) case (sel)
	0 : out <= in0;
	1 : out <= in1;
	2 : out <= in2;
	3 : out <= in3;
	4 : out <= in4;
	5 : out <= in5;
    endcase
  end
endmodule

module mux8of32dp (out,in,sel,clk); 
  output [7:0] out;
  input [31:0] in;
  input [4:0] sel;
  input clk;
  wire [7:0] L8=0;
  wire [15:0] tmp;
  mux4xNp #(16) mxa (tmp, {in[7:0],L8},in[15:0],in[23:8],in[31:16], sel[4:3],clk);
  mux8xNp #(8) mxb (out, tmp[8:1],tmp[9:2],tmp[10:3],tmp[11:4],tmp[12:5],tmp[13:6],tmp[14:7],tmp[15:8], sel[2:0],clk);
endmodule

