/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  2D LUT Based Demod Core Interface

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

*/
module dmodengine (sclk,srst, scs,saddr, swr,swrbus, srd,srdbus, sack,
		   ioclk, istat,iena,isel,ibus, ostat,oena,osel,obus, test);

  parameter PORT=1;	// engine index
  parameter IBW=64;	// input data width
  parameter OBW=16;	// output data width

  localparam SBW=32;	// System bus width - always 32
  localparam DBW=32;	// Data bus width

  input sclk;		// system clock
  input srst;		// system reset
  input scs;		// system select
  input swr;		// register write
  input srd;		// register read
  input [31:0] saddr;	// register address
  input [31:0] swrbus;	// register data
  output [31:0] srdbus;	// register data
  output sack;		// acknowledge

  input ioclk;		// I/O data bus clock
  output istat;		// istat - ready to accept 64 by burst
  input iena;		// input enable (one cycle early) (continuous 64 byte burst)
  input [7:0] isel;	// input channel selection
  input [IBW-1:0] ibus;	// input data bus
  output ostat;		// ostat - ready to provide 64 byte burst
  input oena;		// output enable (one cycle early) (continuous 64 byte burst)
  output [7:0] osel;	// output channel selection
  output [IBW-1:0] obus;// output data bus

  output [7:0] test;	// 8 test signals for scope probes

  wire L=0, H=1;
  assign sack=H;
  assign osel=0;

  // System status ram with parameterized ID 
  swrstat #("DMOD",0) ss (sclk,scs,swr, saddr, swrbus,srdbus);

  // System register enables - with routing pipeline
  wire [8:0] sb; swrbank #(0,9) swb (sclk,scs,swr,saddr,sb);
  wire swrsys  = sb[0]; // system register (required)
  wire swrdec  = sb[1]; // decimation register (optional)
  wire swrrate = sb[3]; // rate register (optional)
  wire swrfram = sb[5]; // frame register (optional
  wire swrflag = sb[7]; // flags register (optional)
  wire swrlut  = sb[8]; // LUT register (optional)

  reg [31:0] system,dec,rate,frame,flag;
  always @(posedge sclk) begin
    if (srst)   system <= 0; else if (swrsys) system <= swrbus;
    if (srst)   dec    <= 0; else if (swrdec) dec    <= swrbus;
    if (srst)   rate   <= 0; else if (swrrate) rate  <= swrbus;
    if (srst)   frame  <= 0; else if (swrfram) frame <= swrbus;
    if (srst)   flag   <= 0; else if (swrflag) flag  <= swrbus;
  end
  wire reset = !system[0];
  wire reduce = system[2];
  wire demod  = system[3];
  wire [3:0] ifmt = system[11:8];
  wire [3:0] ofmt = system[15:12];
  wire throttle = flag[0];
  wire swap = flag[1];

  reg ival,dena,dval,wval;
  wire vstat,wstat,nena,nout,dout;
  wire [DBW-1:0] vbus,dbus,nbus;
  reg [DBW-1:0] wbus;

  wire vena = demod? dena : nena;

  always @(posedge ioclk) ival <= iena;

  always @(posedge sclk) begin
    if (reset) dena <= 0; else dena <= vstat && wstat;
    if (reset) dval <= 0; else dval <= dena;
    if (reset) wval <= 0; else wval <= demod? dout : nout;
                               wbus <= demod? dbus : nbus;
  end

  fifoNxMfmt #(IBW,DBW) fi (ioclk,reset, ifmt,istat,ival,ibus, sclk,reset, 4'h8, vstat,vena,vbus);

  // Noop Engine
  noop np (sclk,reset, reduce,dec,frame,throttle,rate,swap, vstat,nena,vbus, wstat,nout,nbus);

  // Demod Engine
  demod_2dlut dm (sclk,reset, swrlut,swrbus, dval,vbus, dout,dbus);

  fifoNxMfmti #(IBW,DBW) fo (ioclk,reset, ofmt,ostat,oena,obus, sclk,reset, 4'h8, wstat,wval,wbus);

  assign test = {oena,ostat,iena,istat,wval,vena,demod,reset};
 //  assign test = {oena,ostat,iena,istat,swrlut,swrbus[31],swrbus[0],system[0]};

endmodule

/*
 2D LUT Demod

 Resources: 2 mult, 4 bram

 Timing:

 ival  aval
 lut   bval  dvdx dvdy
 mul         v dvx dvy
 out         tv

*/

module demod_2dlut (clk,rst, wlut,dlut, ival,ibus, oval,obus);

  parameter DW=16;	// input data width
  parameter LW=6;       // LUT bits 

  localparam TW=LW*2-2;	// table address bits
  localparam OW=16;      // output width - same as coeff width
  localparam PW=DW-LW+OW;// product width of derivatives
  localparam CDW=DW*2;	// complex input data width
`ifdef VIRTEX2
  localparam RR=0;	// no registered ram
`else
  localparam RR=1;	// registered ram
`endif

  input clk,rst,wlut;
  input [31:0] dlut;	// coefficient LUT data

  input ival;
  input [CDW-1:0] ibus;

  output oval;
  output [CDW-1:0] obus;

  wire L=0, H=1;

  wire signed [DW-1:0] ibusr = $signed(ibus[DW-1:0]);
  wire signed [DW-1:0] ibusi = $signed(ibus[CDW-1:DW]);

  reg [7:0] flags;
  wire diff = flags[0];
  wire quad = flags[1];

  // table load auto address
  reg [11:0] count;
  always @(posedge clk) begin
    if (rst) count <= 0; else if (wlut) count <= count+1;
  end

  // handle quadrant reduction
  wire [1:0] qn = ibus[CDW-1]? (ibus[DW-1]? 2:3) : (ibus[DW-1]? 1:0);
  wire [1:0] qnd; srl16xNp #(2) srl (clk,H,4'h2,qn,qnd);

  // quadrant allocation
  wire [DW-1:0] jbusr,jbusi;
  reg  [DW-1:0] kbusr,kbusi;
  mux4xN #(DW) jbr (jbusr, ibusr,ibusi,-ibusr,-ibusi, qn);
  mux4xN #(DW) jbi (jbusi, ibusi,-ibusr,-ibusi,ibusr, qn);
  always @(posedge clk) kbusr <= jbusr;
  always @(posedge clk) kbusi <= jbusi;

  wire signed [OW-1:0] v,dvdx,dvdy,w;
  wire [8:0] caddr = count[8:0];
  wire [2:0] csel  = count[11:9];
  wire [TW-1:0] alut = {jbusi[DW-2:DW-LW],jbusr[DW-2:DW-LW]};
  wire [TW-1:0] blut = {kbusi[DW-2:DW-LW],kbusr[DW-2:DW-LW]};
  wire [TW-1:0] clut;
  srl16xN #(TW) bsrl (clk,H,4'h2,blut,clut);
  sdpram #(2,32,16,0,0)  rv  (clk,H,caddr,wlut&&(csel==0),dlut,    clk,H,        blut,v);
  sdpram #(2,32,16,0,RR) rvx (clk,H,caddr,wlut&&(csel==1),dlut,    clk,H,RR?alut:blut,dvdx);
  sdpram #(2,32,16,0,RR) rvy (clk,H,caddr,wlut&&(csel==2),dlut,    clk,H,RR?alut:blut,dvdy);
  sdpram #(2,32,16,0,0)  rw  (clk,H,caddr,wlut&&(csel==3),dlut,    clk,H,        clut,w);
  always @(posedge clk) if (wlut&&(csel==4)) flags <= dlut[7:0];

  reg qua,lup,mul,dif,out;
  always @(posedge clk) begin
    qua <= ival;	// determine quadrant
    lup <= qua;		// lookup values
    mul <= lup;		// multiply partials
    dif <= mul;		// apply difference for FM
    out <= dif;		// present output
  end

  reg [DW-LW-1:0] dx,dy;
  //reg signed [PW-1:0] dvx,dvy;
  reg signed [PW:0] dvx,dvy;
  reg signed [OW:0] dv,ev,ltv,fv;
  wire [1:0] qne = ev[OW]? (ev[OW-1]? qnd-1 : qnd+1) : qnd;
  wire signed [OW:0] tv = quad? {L,qne,ev[OW-1:2]} : ev;

  always @(posedge clk) begin
    dx  <= kbusr[DW-LW-1:0];
    dy  <= kbusi[DW-LW-1:0];
    dvx <= dvdx * $signed({L,dx});
    dvy <= dvdy * $signed({L,dy});
    dv  <= {v,L}; // register for speed in next add - move to MSB 
    ev  <= dv + dvx[PW-1:PW-OW-1] + dvy[PW-1:PW-OW-1];
    if (rst) ltv <= 0; else if (dif&diff) ltv <= tv;
    fv <= tv - ltv;
//    if (ival) $write("LUT r=%x i=%x q=%x a=%x v=%x tv=%x\n",jbusr,jbusi,qn,alut,v,tv);
  end
  assign obus = {w,fv[OW-1:0]};
  assign oval = out;

endmodule
